﻿using System.Data.SqlClient;
using System.Linq;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.HL7.DataStructure;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.HibIntegration.Loader
{
    public class HealthProviderIndividualLoader
    {
        #region Private Static Fields

        // Placeholder values for situations when information is unknown or not supplied.
        private const int UnknownForeignKey = -1;

        #endregion Private Static Fields

        #region Private Instance Fields

        private SqlTransaction transaction;
        private UserDetails user;

        #endregion Private Instance Fields

        #region Constructors

        public HealthProviderIndividualLoader(UserDetails user, SqlTransaction transaction)
        {
            this.user = user;
            this.transaction = transaction;
        }

        #endregion Constructors

        #region Public Methods

        /// <summary>
        /// Loads a health provider individual from an HL7 message.
        /// </summary>
        /// <param HL7Name="hospital">Hospital</param>
        /// <param HL7Name="xcn">Health Provider HL7 structure</param>
        public HealthProviderIndividual Load(int hospitalId, XCN xcn)
        {
            string pasIdentifier = xcn.IDnumberST;
            HealthProviderIndividualDl dataAccess = new HealthProviderIndividualDl(user);
            HealthProviderIndividual provider;
            dataAccess.GetByPasIdentifier(hospitalId, pasIdentifier, out provider);
            HealthProviderIndividual originalProvider = CloneProvider(provider);
            if (!provider.HealthProviderIndividualId.HasValue)
            {
                HospitalHealthProviderIndividual link = new HospitalHealthProviderIndividual();
                link.HospitalId = hospitalId;
                link.PasProviderIdentifier = pasIdentifier;
                provider.HospitalHealthProviderIndividuals.Add(link);
            }

            provider.GivenNames = (xcn.givenname + " " + xcn.middleinitialorname).Trim();

            provider.FamilyName = (xcn.familylastname == null) ? string.Empty : xcn.familylastname.familyname;
            ListSingleton lists = ListSingleton.Instance;
            string titleCd = xcn.prefix ?? string.Empty;
            Title title = lists.AllTitles.FirstOrDefault(a => a.Code.ToUpper() == titleCd.ToUpper());
            string suffixCd = xcn.suffix ?? string.Empty;
            Suffix suffix = lists.AllSuffixes.FirstOrDefault(a => a.Code.ToUpper() == suffixCd.ToUpper());
            provider.TitleId = (title == null ? UnknownForeignKey : title.Id.Value);
            provider.SuffixId = (suffix == null ? UnknownForeignKey : suffix.Id.Value);

            if (WasProviderChanged(originalProvider, provider))
            {
                // Save the provider
                if (!dataAccess.Save(provider, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, dataAccess.GetType().FullName));
                }
            }
            return provider;
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Makes a new HealthProviderIndividual object with copies of the key fields that
        /// we need to check to determine whether a change was made that needs to be
        /// applied in the database: GivenNames, FamilyName, TitleId and SuffixId.
        /// </summary>
        /// <param name="provider">The health provider object obtained from the database.</param>
        /// <returns>A new object with copies of the key fields of the supplied object.</returns>
        private HealthProviderIndividual CloneProvider(HealthProviderIndividual provider)
        {
            HealthProviderIndividual clone = new HealthProviderIndividual();
            clone.GivenNames = provider.GivenNames;
            clone.FamilyName = provider.FamilyName;
            clone.TitleId = provider.TitleId;
            clone.SuffixId = provider.SuffixId;
            return clone;
        }

        /// <summary>
        /// Determines whether a change was made to the HealthProviderIndividual that needs to be
        /// applied in the database.
        /// </summary>
        /// <param name="originalProvider">The object retured from the CloneProvider method.</param>
        /// <param name="changedProvider">The object obtained from the database, after populating information from the HL7 message.</param>
        /// <returns>Whether the object needs to be saved.</returns>
        private bool WasProviderChanged(HealthProviderIndividual originalProvider, HealthProviderIndividual changedProvider)
        {
            return originalProvider.FamilyName != changedProvider.FamilyName
                || originalProvider.GivenNames != changedProvider.GivenNames
                || originalProvider.TitleId != changedProvider.TitleId
                || originalProvider.SuffixId != changedProvider.SuffixId;
        }

        #endregion Private Methods
    }
}