﻿using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.HibIntegration.HL7.DataStructure;
using HipsSchema = HIPS.PcehrDataStore.Schemas;

namespace HIPS.HibIntegration.Loader
{
    /// <summary>
    /// Handles merge and move messages from PAS and EMPI.
    /// </summary>
    public class MergeLoader
    {
        private SqlTransaction transaction;

        public MergeLoader(CommonSchemas.UserDetails user, SqlTransaction transaction)
        {
            this.User = user;
            this.transaction = transaction;
        }

        private static string[] SupportedTriggerEvents
        {
            get
            {
                return new string[]
                {
                    TriggerEventResource.MergeEnterpriseNumbers,
                    TriggerEventResource.MergeMrns,
                    TriggerEventResource.MergeVisitNumbers,
                    TriggerEventResource.MoveMrnToEnterpriseNumber,
                    TriggerEventResource.MoveVisitInformation,
                    TriggerEventResource.MoveVisitToAnotherPatient
                };
            }
        }

        private CommonSchemas.UserDetails User
        {
            get;
            set;
        }

        /// <summary>
        /// Determines whether or not this class can process a given HL7 message.
        /// </summary>
        /// <param name="message"></param>
        /// <returns></returns>
        internal static bool CanProcess(HL7.Message.HL7GenericPasMessage message)
        {
            return SupportedTriggerEvents.Contains(message.MessageHeader.MessageType.triggerevent);
        }

        /// <summary>
        /// Processes an HL7 merge or move message.
        /// </summary>
        /// <param name="message">The HL7 message</param>
        /// <param name="mrn">The MRN.</param>
        /// <param name="statePatientId">The state/territory patient identifier.</param>
        /// <param name="patientMaster">Returns the surviving or destination patient master.</param>
        /// <returns>Whether IHI validation should be done on the returned patient master.</returns>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageInfoException">When the message is to be skipped because the source information is not found.</exception>
        internal bool Process(HL7.Message.HL7GenericPasMessage message, CX mrn, CX statePatientId, out HipsSchema.Hospital hospital, out HipsSchema.PatientMaster patientMaster, out HipsSchema.PatientMaster oldPatientMaster)
        {
            hospital = FindHospital(mrn);
            bool mergeNeedsIhiValidation;
            string trigger = message.MessageHeader.MessageType.triggerevent;
            if (trigger == TriggerEventResource.MergeEnterpriseNumbers)
            {
                MergeEnterpriseNumbers(message, statePatientId, hospital, out patientMaster);
                oldPatientMaster = null;
                mergeNeedsIhiValidation = true;
            }
            else if (trigger == TriggerEventResource.MergeMrns)
            {
                // Always validate the IHI after merging two MRNs
                MergeMrns(message, mrn, statePatientId, out patientMaster, out oldPatientMaster);
                mergeNeedsIhiValidation = true;
            }
            else if (trigger == TriggerEventResource.MergeVisitNumbers)
            {
                MergeVisitNumbers(message, mrn, statePatientId);
                patientMaster = null;
                oldPatientMaster = null;
                hospital = null;
                mergeNeedsIhiValidation = false;
            }
            else if (trigger == TriggerEventResource.MoveMrnToEnterpriseNumber)
            {
                // Always validate the IHI after moving an MRN onto a SAUHI.
                MoveMrnToEnterpriseNumber(message, mrn, statePatientId, hospital, out patientMaster, out oldPatientMaster);
                mergeNeedsIhiValidation = true;
            }
            else if (trigger == TriggerEventResource.MoveVisitInformation
                || trigger == TriggerEventResource.MoveVisitToAnotherPatient)
            {
                MoveVisitNumber(message, mrn, statePatientId);
                mergeNeedsIhiValidation = false;
                patientMaster = null;
                oldPatientMaster = null;
                hospital = null;
            }
            else
            {
                string error = string.Format(ConstantsResource.Unhandled_HL7_Message_Type, trigger);
                throw new HL7MessageErrorException(error);
            }
            return mergeNeedsIhiValidation;
        }

        /// <summary>
        /// Extracts the visit number from the move visit message. According to
        /// Standard Enterprise HL7 Specification 3.0 document, the EPAS A45
        /// messages will have the visit number in MRG-5 Prior Visit Number.
        /// However the legacy PAS systems send an A51 message with the visit
        /// number in PV1-19 Visit Number.
        /// </summary>
        /// <param name="message">The move visit message.</param>
        /// <returns>The visit number</returns>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageErrorException">If no visit number is found.</exception>
        private static string ExtractVisitNumberForMoveVisit(HL7.Message.HL7GenericPasMessage message)
        {
            string visitNum;
            if (message.MergeInformation != null && message.MergeInformation.PriorVisitNumber != null)
            {
                visitNum = message.MergeInformation.PriorVisitNumber.ID;
            }
            else if (message.PatientVisit != null && message.PatientVisit.VisitNumber != null)
            {
                visitNum = message.PatientVisit.VisitNumber.ID;
            }
            else
            {
                string error = string.Format(ConstantsResource.NoVisitNumberForEpisode);
                throw new HL7MessageErrorException(error);
            }
            return visitNum;
        }

        /// <summary>
        /// Finds the hospital that is the assigning authority for the MRN.
        /// </summary>
        /// <param name="mrn">The MRN</param>
        /// <returns>The hospital</returns>
        private static HipsSchema.Hospital FindHospital(CX mrn)
        {
            if (mrn == null || mrn.assigningauthority == null)
            {
                throw new HL7MessageErrorException(ConstantsResource.NoMrnForPatient);
            }
            HipsSchema.Hospital hospital;
            string hospitalCode = mrn.assigningauthority.namespaceID;
            string hospitalCodeSystem = ConstantsResource.PatientAdministrationSystemHospitalCodeSystem;
            hospital = HospitalSingleton.Value.Find(hospitalCode, hospitalCodeSystem);
            if (hospital == null)
            {
                string error = string.Format(ConstantsResource.CannotFindHospital, hospitalCode, hospitalCodeSystem);
                throw new HL7MessageErrorException(error);
            }
            return hospital;
        }

        /// <summary>
        ///            1.	Locate the surviving and non-surviving patient masters using the SAUHIs.
        ///            2.	If only the non-surviving patient master exists, just change its SAUHI.
        ///            3.	If only the surviving patient master exists, do nothing.
        ///            4.	If neither patient master exists, do nothing.
        ///            5.	If both patient masters exist:
        ///                a)	Set the non-surviving patient master to inactive.
        ///                b)	Transfer all hospital patient records from the
        ///                     non-surviving patient master to the surviving
        ///                     patient master.
        ///                c)	If neither patient master has an IHI:
        ///                    -	Attempt IHI search with demographics of the surviving patient master.
        ///                    -	If successful, assign the IHI to the surviving patient master.
        ///                d)	If each had the same IHI, or only one had an IHI:
        ///                    -	If either of the IHIs had a status of “Duplicated IHI”, or “Potential Duplicate Patient”, then remove the alert, as it has now been resolved.
        ///                    -	Validate this IHI with the surviving patient master’s demographics.
        ///                    -	If successful, assign the IHI to the surviving patient master.
        ///                e)	If the two patient masters have different IHIs:
        ///                    -	Create alert against both IHIs, so that they cannot be used until support staff have selected the correct one to be associated with the merged patient health record.
        ///                    -	The HI service should be notified of a potential duplicate IHI (two different IHIs for the same person).
        /// </summary>
        /// <param name="message">The parsed A34 enterprise merge message</param>
        /// <param name="statePatientId">The enterprise identifier that survives after the merge is completed.</param>
        /// <param name="survivingPatientMaster">Returns the patient master that survives after the merge is completed.</param>
        private void MergeEnterpriseNumbers(HL7.Message.HL7GenericPasMessage message, CX statePatientId, HipsSchema.Hospital hospital, out HipsSchema.PatientMaster survivingPatientMaster)
        {
            PatientAccess patientAccess = new PatientAccess(User);
            HipsSchema.PatientMaster nonSurvivingPatientMaster;
            string oldId = message.MergeInformation.PriorPatientID.ID;
            string newId = statePatientId.ID;
            patientAccess.PatientMasterDataAccess.GetByStatePatientId(newId, hospital.HealthProviderOrganisationNetworkId, out survivingPatientMaster);
            patientAccess.PatientMasterDataAccess.GetByStatePatientId(oldId, hospital.HealthProviderOrganisationNetworkId, out nonSurvivingPatientMaster);

            if (survivingPatientMaster == null || !survivingPatientMaster.PatientMasterId.HasValue)
            {
                if (nonSurvivingPatientMaster == null || !nonSurvivingPatientMaster.PatientMasterId.HasValue)
                {
                    // Neither patient master exists, do nothing.
                    string info = string.Format(ConstantsResource.EnterpriseMergeNeitherFound, oldId, newId);
                    throw new HL7MessageInfoException(info);
                }
                else
                {
                    //  only the non-surviving patient master exists, just change its SAUHI.
                    nonSurvivingPatientMaster.StatePatientId = statePatientId.ID;
                    if (!patientAccess.PatientMasterDataAccess.Update(nonSurvivingPatientMaster, transaction))
                    {
                        string info = string.Format(ConstantsResource.EnterpriseMergeToNewIdFailed, oldId, newId);
                        throw new HL7MessageErrorException(info);
                    }
                    survivingPatientMaster = nonSurvivingPatientMaster;
                }
            }
            else
            {
                if (nonSurvivingPatientMaster == null || !nonSurvivingPatientMaster.PatientMasterId.HasValue)
                {
                    // Only the surviving patient master exists, do nothing.
                    string info = string.Format(ConstantsResource.EnterpriseMergeOnlySurvivorFound, oldId, newId);
                    throw new HL7MessageInfoException(info);
                }
                else
                {
                    // Both patient masters exist.
                    if (!patientAccess.PatientMasterDataAccess.Merge(nonSurvivingPatientMaster, survivingPatientMaster, hospital.HealthProviderOrganisationNetworkId, transaction))
                    {
                        string info = string.Format(ConstantsResource.EnterpriseMergeFailed, oldId, newId);
                        throw new HL7MessageErrorException(info);
                    }
                }
            }
        }

        /// <summary>
        /// 1.	Locate the surviving and non-surviving hospital patient records
        ///     by MRN and facility code.
        /// 2.  If the non-surviving hospital patient is not found, skip the
        ///     message.
        /// 3.  If the surviving hospital patient is not found:
        ///        a. Change the MRN in the non-surviving hospital patient to
        ///           the survivor's MRN.
        ///     Otherwise:
        ///        a.  Move the non-surviving hospital patient and all
        ///            previously merged MRNs on the same patient master onto
        ///            the patient master of the surviving hospital patient.
        ///        b.  Move all episodes from the non-surviving hospital
        ///            patient to the surviving hospital patient.
        ///        c.  Move all pathology results from the non-surviving
        ///            hospital patient, to the surviving hospital patient.
        /// </summary>
        /// <param name="message">The merge MRNs HL7 message (A36)</param>
        /// <param name="toMrn">The MRN of the surviving hospital patient</param>
        /// <param name="statePatientId">The enterprise ID of the surviving hospital patient</param>
        /// <param name="toPM">Returns the patient master of the surviving hospital patient.</param>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageInfoException">When the non-surving hospital patient is not found.</exception>
        private void MergeMrns(HL7.Message.HL7GenericPasMessage message, CX toMrn, CX statePatientId, out HipsSchema.PatientMaster toPM, out HipsSchema.PatientMaster fromPM)
        {
            HipsSchema.Hospital hospital;
            PatientAccess patientAccess = new PatientAccess(User);
            HipsSchema.HospitalPatient toHP, fromHP;
            CX fromMrn = PatientLoader.IdentifierWithType(
                message.MergeInformation.PriorPatientIdentifierList,
                Enumerators.IdentifierTypeCode.MR);
            PatientLinker linker = new PatientLinker(User, transaction);
            patientAccess.HospitalPatientDataAccess.Get(FindHospital(fromMrn).HospitalId.Value, fromMrn.ID, out fromHP);
            int originalPatientMasterId = fromHP.PatientMasterId;

            // Find and link the old MRN with the enterprise ID (e.g. SAUHI).
            // If there is an enterprise ID in the message, this is the point
            // at which the fromHP moves onto the same PM as the toHP.
            // Otherwise it happens in PerformMrnMerge below.
            linker.FindAndLinkPatient(fromMrn, statePatientId, out hospital, out fromHP, out fromPM);

            if (!fromHP.PatientId.HasValue)
            {
                // The source MRN does not exist, skip this message.

                string info = string.Format(ConstantsResource.MergeMrnSourceNotFound, fromMrn.ID, toMrn.ID);
                throw new HL7MessageInfoException(info);
            }

            // Find and link the new MRN with the SAUHI - usually there will be no movement on this one.
            linker.FindAndLinkPatient(toMrn, statePatientId, out hospital, out toHP, out toPM);

            if (!toHP.PatientId.HasValue)
            {
                // Only the old MRN existed. No need to save the new Hospital
                // Patient and move everything across: for simplicity just
                // change the MRN on the old one and update that.
                fromHP.Mrn = toMrn.ID;
                if (!patientAccess.HospitalPatientDataAccess.Update(fromHP, transaction))
                {
                    string error = string.Format(ConstantsResource.DatabaseError, patientAccess.HospitalPatientDataAccess.GetType().FullName);
                    throw new HL7MessageErrorException(error);
                }
                toPM = fromPM;
                fromPM = null;
            }
            else
            {
                // Both MRNs existed. There is no need to mark which MRN is active:
                // the lack of any episodes on the non-survivor will suppress its use.
                PerformMrnMerge(patientAccess, toHP, fromHP, toPM);

                // Check whether fromPM has any HPs remaining - if not, then no need to revalidate IHI.
                List<HipsSchema.HospitalPatient> remaining = patientAccess.HospitalPatientDataAccess.GetAll(null, originalPatientMasterId, transaction);
                if (remaining.Count == 0)
                {
                    fromPM = null;
                }
                else
                {
                    patientAccess.PatientMasterDataAccess.Get(originalPatientMasterId, hospital.HealthProviderOrganisationNetworkId, out fromPM, transaction);
                }
            }
        }

        /// <summary>
        /// 1.	Locate the hospital patient by MRN and facility code. If the
        ///     hospital patient does not exist, skip the message.
        /// 2.	Within the patient, locate the non-surviving episode by visit
        ///     number. If the episode does not exist, skip the message.
        /// 3.  If the surviving episode does not exist:
        ///        Change the visit number on the non-surviving episode to the
        ///        surviving visit number.
        ///     Otherwise, if both episodes exist:
        ///        a. Set the non-surviving episode to inactive.
        ///        b. Update clinical documents attached to the non-surviving episode,
        ///           attaching them to the surviving episode.
        ///        c. If the non-surviving episode had consent withdrawn, then set the
        ///           withdrawal of consent on the surviving episode to withdrawn.
        /// </summary>
        /// <param name="message">The merge visit numbers HL7 message (A35).</param>
        /// <param name="mrn">The patient's MRN</param>
        /// <param name="statePatientId">The patient's enterprise ID.</param>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageInfoException">When the MRN or non-surviving episode are not found.</exception>
        private void MergeVisitNumbers(HL7.Message.HL7GenericPasMessage message, CX mrn, CX statePatientId)
        {
            HipsSchema.Hospital hospital;
            HipsSchema.HospitalPatient hospitalPatient;
            HipsSchema.PatientMaster patientMaster;
            PatientAccess patientAccess = new PatientAccess(User);
            PatientLinker linker = new PatientLinker(User, transaction);
            string oldVisitNum = message.MergeInformation.PriorVisitNumber.ID;
            string newVisitNum = message.PatientVisit.VisitNumber.ID;

            linker.FindAndLinkPatient(mrn, statePatientId, out hospital, out hospitalPatient, out patientMaster);
            if (!hospitalPatient.PatientId.HasValue)
            {
                // The source patient does not exist, skip this message.
                string info = string.Format(ConstantsResource.MergeVisitSourcePatientNotFound, oldVisitNum, newVisitNum, mrn.ID);
                throw new HL7MessageInfoException(info);
            }

            //Specific Episode must be found from the Visit Number in this case
            HipsSchema.Episode survivingEpisode = patientAccess.GetEpisode(newVisitNum, hospitalPatient);

            //Specific Episode must be found from the Visit Number in this case
            HipsSchema.Episode nonSurvivingEpisode = patientAccess.GetEpisode(oldVisitNum, hospitalPatient);

            if (nonSurvivingEpisode == null)
            {
                // The source episode does not exist, skip this message.
                string info = string.Format(ConstantsResource.MergeVisitSourceEpisodeNotFound, oldVisitNum, newVisitNum, mrn.ID);
                throw new HL7MessageInfoException(info);
            }

            if (survivingEpisode == null)
            {
                // We only have the non-surviving episode. Change its visit number and be done with it.
                nonSurvivingEpisode.SourceSystemEpisodeId = message.PatientVisit.VisitNumber.ID;
                if (!patientAccess.EpisodeDataAccess.Update(nonSurvivingEpisode, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.EpisodeDataAccess.GetType().FullName));
                }
            }
            else
            {
                List<HipsSchema.ClinicalDocument> clinicalDocuments = patientAccess.DocumentDataAccess.GetAll(nonSurvivingEpisode.EpisodeId.Value, null);
                foreach (HipsSchema.ClinicalDocument document in clinicalDocuments)
                {
                    document.EpisodeId = survivingEpisode.EpisodeId.Value;
                    if (!patientAccess.DocumentDataAccess.Save(document, transaction))
                    {
                        throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.DocumentDataAccess.GetType().FullName));
                    }
                }
                if (nonSurvivingEpisode.ConsentWithdrawn)
                {
                    survivingEpisode.ConsentWithdrawn = true;
                    if (!patientAccess.EpisodeDataAccess.Update(survivingEpisode, transaction))
                    {
                        throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.EpisodeDataAccess.GetType().FullName));
                    }
                }
                nonSurvivingEpisode.EpisodeLifecycleId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.EpisodeLifecycle.Merged;
                if (!patientAccess.EpisodeDataAccess.Update(nonSurvivingEpisode, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.EpisodeDataAccess.GetType().FullName));
                }
            }
        }

        /// <summary>
        /// Processes an HL7 message that moves an MRN from one SAUHI to another.
        /// If the MRN is not found then the message is skipped.
        /// If the destination SAUHI is not found then the MRN is unlinked to a new patient master.
        /// If the destination SAUHI is found then the MRN is moved to that patient master.
        /// All previously merged MRNs from the same hospital as the moving MRN move with it.
        /// The destination patient master is always returned for IHI validation.
        /// If there are any MRNs left on the source patient master, then it is returned for IHI validation.
        /// </summary>
        /// <param name="message">The incoming ADT^A43 message.</param>
        /// <param name="mrn">The MRN that is moving</param>
        /// <param name="statePatientId">The SAUHI that the MRN is moving to</param>
        /// <param name="hospital">The hospital at which the MRN is based</param>
        /// <param name="destinationPatientMaster">Returns the patient master to which the MRN moved</param>
        /// <param name="sourcePatientMaster">Returns the patient master from which the MRN moved, but only if there are any MRNs left on it.</param>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageInfoException">If the source MRN is not found.</exception>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageErrorException">If there is an error accessing the database.</exception>
        private void MoveMrnToEnterpriseNumber(HL7.Message.HL7GenericPasMessage message, CX mrn, CX statePatientId, HipsSchema.Hospital hospital, out HipsSchema.PatientMaster destinationPatientMaster, out HipsSchema.PatientMaster sourcePatientMaster)
        {
            PatientAccess patientAccess = new PatientAccess(User);
            string hospitalCode = mrn.assigningauthority.namespaceID;
            string hospitalCodeSystem = ConstantsResource.PatientAdministrationSystemHospitalCodeSystem;
            HipsSchema.HospitalPatient sourceHospitalPatient;

            Mrn sourceMrn = new Mrn(mrn.ID, hospitalCode, hospitalCodeSystem);
            HipsResponse sourceHipsResponse = patientAccess.GetPatient(sourceMrn, hospital, out sourceHospitalPatient, out sourcePatientMaster);

            if (sourceHipsResponse.Status != HipsResponseIndicator.OK)
            {
                throw new HL7MessageErrorException(sourceHipsResponse.HipsErrorMessage);
            }

            if (sourceHospitalPatient == null)
            {
                // The source patient does not exist, skip this message.
                string info = string.Format(ConstantsResource.MoveMrnSourceNotFound, mrn.ID, statePatientId.ID);
                throw new HL7MessageInfoException(info);
            }
            else
            {
                HipsResponse destinationHipsResponse = patientAccess.PatientMasterDataAccess.GetByStatePatientId(statePatientId.ID, hospital.HealthProviderOrganisationNetworkId, out destinationPatientMaster);
                if (destinationHipsResponse.Status == HipsResponseIndicator.DatabaseError)
                {
                    throw new HL7MessageErrorException(destinationHipsResponse.HipsErrorMessage);
                }
                else if (destinationHipsResponse.Status == HipsResponseIndicator.OK)
                {
                    // HIPS has an existing patient master with the destination SAUHI.
                    HipsResponse response = patientAccess.HospitalPatientDataAccess.Move(sourceHospitalPatient, destinationPatientMaster, transaction);
                    if (response.Status != HipsResponseIndicator.OK)
                    {
                        throw new HL7MessageErrorException(response.HipsErrorMessage);
                    }

                    // No demographic update happens on a typical A43.
                }
                else
                {
                    // Unlink to new patient master.
                    destinationPatientMaster = sourcePatientMaster.CloneToNewStatePatientId(statePatientId.ID);
                    if (!patientAccess.PatientMasterDataAccess.Insert(destinationPatientMaster, transaction))
                    {
                        throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.HospitalPatientDataAccess.GetType().FullName));
                    }

                    HipsResponse response = patientAccess.HospitalPatientDataAccess.Move(sourceHospitalPatient, destinationPatientMaster, transaction);
                    if (response.Status != HipsResponseIndicator.OK)
                    {
                        throw new HL7MessageErrorException(response.HipsErrorMessage);
                    }
                }
            }

            // Check whether fromPM has any HPs remaining (from other hospitals). If not, then no need to revalidate IHI.
            List<HipsSchema.HospitalPatient> remaining = patientAccess.HospitalPatientDataAccess.GetAll(null, sourcePatientMaster.PatientMasterId.Value, transaction);
            if (remaining.Count == 0)
            {
                sourcePatientMaster = null;
            }
        }

        /// <summary>
        /// Moves an episode from one hospital patient to another.
        /// This can handle both A45 messages and A51 messages.
        /// A45 messages lack a PV1 segment and have the visit number in MRG-5 Prior Visit Number.
        /// A51 messages have the visit number in PV1-19 Visit Number.
        ///
        /// 1.	Locate the source hospital patient and destination hospital patient
        ///     by MRN and facility code. If the source or destination hospital
        ///     patient do not exist, create the patient using the demographics
        ///     provided in the HL7 message.
        /// 2.	Locate the moving episode under the source hospital patient
        ///     using the visit number. If the moving episode does not exist
        ///     then create the episode using the information provided in the
        ///     HL7 message.
        /// 3.	Attach the moving episode to the destination hospital patient.
        /// </summary>
        /// <param name="message">The move visit message (A45 or A51).</param>
        /// <param name="mrn">The destination MRN</param>
        /// <param name="statePatientId">The destination enterprise ID.</param>
        private void MoveVisitNumber(HL7.Message.HL7GenericPasMessage message, CX mrn, CX statePatientId)
        {
            PatientAccess patientAccess = new PatientAccess(User);
            PatientLinker linker = new PatientLinker(User, transaction);
            string trigger = message.MessageHeader.MessageType.triggerevent;
            CX sourceId = PatientLoader.IdentifierWithType(
                message.MergeInformation.PriorPatientIdentifierList,
                Enumerators.IdentifierTypeCode.MR,
                message.MergeInformation.PriorPatientID);
            string visitNum = ExtractVisitNumberForMoveVisit(message);

            HipsSchema.Hospital hospital;
            HipsSchema.HospitalPatient sourceHospitalPatient, destinationHospitalPatient;
            HipsSchema.PatientMaster sourcePatientMaster, destinationPatientMaster;
            linker.FindAndLinkPatient(sourceId, null, out hospital, out sourceHospitalPatient, out sourcePatientMaster);

            if (!sourceHospitalPatient.PatientId.HasValue)
            {
                // The source patient does not exist, skip this message.
                string info = string.Format(ConstantsResource.MoveVisitSourcePatientNotFound, visitNum, sourceId.ID, mrn.ID);
                throw new HL7MessageInfoException(info);
            }

            //Specific Episode must be found from the Visit Number in this case
            HipsSchema.Episode episode = patientAccess.GetEpisode(visitNum, sourceHospitalPatient);
            if (episode != null)
            {
                linker.FindAndLinkPatient(mrn, statePatientId, out hospital, out destinationHospitalPatient, out destinationPatientMaster);
                if (!destinationHospitalPatient.PatientId.HasValue)
                {
                    // The destination patient does not exist, create it
                    PatientLoader loader = new PatientLoader(User, transaction);
                    HipsSchema.PatientMaster oldPatientMaster;
                    loader.Populate(message.PatientIdentification, hospital, destinationHospitalPatient, destinationPatientMaster, out oldPatientMaster);
                }

                episode.PatientId = destinationHospitalPatient.PatientId.Value;
                if (!patientAccess.EpisodeDataAccess.Update(episode, transaction))
                {
                    string error = string.Format(ConstantsResource.MoveVisitFailed, visitNum, sourceId.ID, mrn.ID);
                    throw new HL7MessageErrorException(error);
                }
            }
            else
            {
                // The episode does not exist, skip this message
                string info = string.Format(ConstantsResource.MoveVisitEpisodeNotFound, visitNum, sourceId.ID, mrn.ID);
                throw new HL7MessageInfoException(info);
            }
        }

        /// <summary>
        /// Performs the MRN merge. Moves the non-survivor onto the survivor's
        /// patient master, updates the surviving and non-surviving hospital
        /// patients, and moves all episodes and pathology tests from the
        /// non-survivor to the survivor.
        /// </summary>
        /// <param name="patientAccess">The patient access helper.</param>
        /// <param name="toHP">The surviving hospital patient.</param>
        /// <param name="fromHP">The non-surviving hospital patient.</param>
        /// <param name="toPM">The surviving patient master.</param>
        /// <exception cref="HIPS.HibIntegration.Loader.HL7MessageErrorException">When database error occurs.</exception>
        private void PerformMrnMerge(PatientAccess patientAccess, HipsSchema.HospitalPatient toHP, HipsSchema.HospitalPatient fromHP, HipsSchema.PatientMaster toPM)
        {
            if (toHP.PatientMasterId != fromHP.PatientMasterId)
            {
                // If there was no SAUHI in the message, then the PatientLinker
                // won't have moved the fromHP onto the toPM by this time, and
                // it still needs to be done.
                HipsResponse response = patientAccess.HospitalPatientDataAccess.Move(fromHP, toPM, transaction);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    throw new HL7MessageErrorException(response.HipsErrorMessage);
                }
            }

            // Update both the survivor and non-survivor in case they changed
            // patient master associations in either the FindAndLinkPatient call
            // or the Move call above.
            if (!patientAccess.HospitalPatientDataAccess.Update(toHP, transaction))
            {
                throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.HospitalPatientDataAccess.GetType().FullName));
            }
            if (!patientAccess.HospitalPatientDataAccess.Update(fromHP, transaction))
            {
                throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.HospitalPatientDataAccess.GetType().FullName));
            }

            // Move the episodes from the non-survivor to the survivor.
            // All Episode LifeCycle Types can be used in this case
            List<HipsSchema.Episode> episodes = patientAccess.EpisodeDataAccess.GetAll(fromHP.PatientId.Value, null);
            foreach (HipsSchema.Episode episode in episodes)
            {
                episode.PatientId = toHP.PatientId.Value;
                if (!patientAccess.EpisodeDataAccess.Update(episode, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.EpisodeDataAccess.GetType().FullName));
                }
            }
        }
    }
}