﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the Contact table
    /// </summary>
    public class ContactDl : BaseDl
    {
        #region Methods

        /// <summary>
        /// Gets a single contact.
        /// </summary>
        /// <param name="contactId">The contact id</param>
        /// <returns>The contact</returns>
        public Contact Get(int contactId)
        {
            Contact result = new Contact();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ContactGet"))
                {
                    AddIntParameter("@ContactId", contactId, command);
                    PopulateBusinessObject<Contact>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageContactGet, ex, User, LogMessage.HIPS_MESSAGE_006);
            }
            return result;
        }

        /// <summary>
        /// Gets all contacts for a patient.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="transaction">Optional transaction</param>
        /// <returns>List of contacts</returns>
        public List<Contact> GetAllByPatient(int patientMasterId, SqlTransaction transaction = null)
        {
            List<Contact> results = new List<Contact>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterContactGet", transaction))
                {
                    command.Parameters.Add(new SqlParameter("@PatientMasterId", patientMasterId));
                    results = GetPopulatedBusinessList<Contact>(command.ExecuteReader());
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorContactGetAllByPatient, ex, User, LogMessage.HIPS_MESSAGE_007);
            }
            return results;
        }

        /// <summary>
        /// Gets all contacts for a hospital. If no hospital ID is provided, gets all contacts that belong to hospitals (but not contacts that belong to patients).
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <returns>List of contacts</returns>
        public List<Contact> GetAllByHospital(int? hospitalId)
        {
            List<Contact> results = new List<Contact>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalContactGet"))
                {
                    command.Parameters.Add(new SqlParameter("@HospitalId", hospitalId));
                    results = GetPopulatedBusinessList<Contact>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorContactGetAllByHospital, ex, User, LogMessage.HIPS_MESSAGE_132);
            }
            return results;
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool Insert(Contact item, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ContactInsert", transaction))
            {
                return base.Insert<Contact>(item, command);
            }
        }

        /// <summary>
        /// Updates the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool Update(Contact item, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ContactUpdate", transaction))
            {
                return base.Update<Contact>(item, command);
            }
        }

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool Delete(Contact item, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ContactDelete", transaction))
            {
                return base.Delete<Contact>(item, command);
            }
        }

        /// <summary>
        /// Updates the specified contacts for a patient.
        /// </summary>
        /// <param name="contacts">The contacts.</param>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool Update(List<Contact> contacts, int patientMasterId, SqlTransaction transaction)
        {
            if (contacts.Count == 0)
            {
                return true;
            }
            bool updateResult = true;
            List<Contact> originalContacts = GetAllByPatient(patientMasterId, transaction);
            List<Contact> newContacts = contacts.FindAll(result => result.ContactId == null);
#if PUMA_CLIENT
            List<Contact> modifiedContacts = contacts.FindAll(result => result.ContactId != null && result.IsDirty == true);
#else
            List<Contact> modifiedContacts = contacts.FindAll(result => result.ContactId != null);
#endif

            IEnumerable<int?> originalIds = from result in originalContacts
                                            select result.ContactId;
            IEnumerable<int?> currentIds = from result in contacts
                                           select result.ContactId;
            IEnumerable<int?> newIds = from result in newContacts
                                       select result.ContactId;
            IEnumerable<int?> deletedIds = originalIds.Except(currentIds);

            updateResult = DeleteContacts(transaction, updateResult, originalContacts, deletedIds);

            if (updateResult)
            {
                updateResult = InsertContacts(patientMasterId, transaction, updateResult, newContacts);
            }
            if (updateResult)
            {
                updateResult = UpdateContacts(patientMasterId, transaction, updateResult, modifiedContacts);
            }
            return updateResult;
        }

        /// <summary>
        /// Inserts the new contacts for a patient.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="updatedContacts">The new contacts.</param>
        /// <returns></returns>
        private bool InsertContacts(int patientMasterId, SqlTransaction transaction, bool updateResult, List<Contact> newContacts)
        {
            foreach (Contact contact in newContacts)
            {
                updateResult = Insert(contact, transaction);
                if (updateResult)
                {
                    updateResult = InsertPatientLink(patientMasterId, contact.ContactId.Value, transaction);
                }
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Updates the contacts for a patient.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="updatedContacts">The updated contacts.</param>
        /// <returns></returns>
        private bool UpdateContacts(int patientMasterId, SqlTransaction transaction, bool updateResult, List<Contact> updatedContacts)
        {
            foreach (Contact contact in updatedContacts)
            {
                updateResult = Update(contact, transaction);
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Deletes the removed contacts.
        /// </summary>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="originalContacts">The original contacts.</param>
        /// <param name="deletedIds">The deleted ids.</param>
        /// <returns></returns>
        private bool DeleteContacts(SqlTransaction transaction, bool updateResult, List<Contact> originalContacts, IEnumerable<int?> deletedIds)
        {
            foreach (int deletedId in deletedIds)
            {
                updateResult = Delete(originalContacts.Find(result => result.ContactId == deletedId), transaction);
                if (updateResult)
                {
                    // Cascading delete will remove the patient link automatically.
                    //updateResult = DeletePatientLink(patientMasterId, deletedId, connection, transaction);
                }
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Inserts the patient link.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="contactId">The contact id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool InsertPatientLink(int patientMasterId, int contactId, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.PatientMasterContactInsert", transaction))
            {
                PatientMasterContact item = new PatientMasterContact(patientMasterId, contactId);
                return base.Insert<PatientMasterContact>(item, command);
            }
        }

        /// <summary>
        /// Deletes the patient link.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="contactId">The contact id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        //public bool DeletePatientLink(int patientMasterId, int contactId, SqlTransaction transaction)
        //{
        //    using (SqlCommand command = GetSqlCommand("hips.PatientMasterContactDelete", transaction))
        //    {
        //        PatientMasterAddress item = new PatientMasterAddress(patientMasterId, contactId);
        //        return base.Delete<PatientMasterAddress>(item, command);
        //    }
        //}

        #endregion Methods
    }
}