﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrDataStore.Schemas.Schemas;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the Hospital table
    /// </summary>
    public class HospitalDl : BaseDl
    {
        #region Private Members

        private ContactDl contactAccess;
        private HospitalAddressDl hospitalAddressAccess;
        private HospitalCodeDl hospitalCodeAccess;

        #endregion Private Members

        #region Properties

        /// <summary>
        /// Gets the contact access.
        /// </summary>
        /// <value>
        /// The contact access.
        /// </value>
        private ContactDl ContactAccess
        {
            get
            {
                if (contactAccess == null)
                {
                    contactAccess = new ContactDl();
                }
                return contactAccess;
            }
        }

        /// <summary>
        /// Gets the hospital address access.
        /// </summary>
        /// <value>
        /// The hospital address access.
        /// </value>
        private HospitalAddressDl HospitalAddressAccess
        {
            get
            {
                if (hospitalAddressAccess == null)
                {
                    hospitalAddressAccess = new HospitalAddressDl();
                }
                return hospitalAddressAccess;
            }
        }

        /// <summary>
        /// Gets the hospital code access.
        /// </summary>
        private HospitalCodeDl HospitalCodeAccess
        {
            get
            {
                if (hospitalCodeAccess == null)
                {
                    hospitalCodeAccess = new HospitalCodeDl();
                }
                return hospitalCodeAccess;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets the specified hospital id.
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <returns></returns>
        public Hospital Get(int hospitalId)
        {
            Hospital result = new Hospital();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalGet"))
                {
                    command.Parameters.Add(new SqlParameter("@HospitalId", hospitalId));
                    PopulateBusinessObject<Hospital>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }

                // Get the hospital addresses, codes and contacts
                result.Addresses = HospitalAddressAccess.GetAll(result.HospitalId);
                result.Codes = HospitalCodeAccess.GetAll(result.HospitalId);
                result.Contacts = ContactAccess.GetAllByHospital(result.HospitalId);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHospitalGet, ex, User, LogMessage.HIPS_MESSAGE_064);
            }
            return result;
        }

        /// <summary>
        /// Gets the specified hospital from the Code.
        /// </summary>
        /// <param name="hospitalId">The hospital code.</param>
        /// <returns></returns>
        public Hospital Get(int codeSystemId, string hospitalCode)
        {
            Hospital result = new Hospital();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalGet"))
                {
                    command.Parameters.Add(new SqlParameter("@Code", hospitalCode));
                    command.Parameters.Add(new SqlParameter("@CodeSystemId", (int)codeSystemId));
                    PopulateBusinessObject<Hospital>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }

                // Get the hospital addresses, codes and contacts
                result.Addresses = HospitalAddressAccess.GetAll(result.HospitalId);
                result.Codes = HospitalCodeAccess.GetAll(result.HospitalId);
                result.Contacts = ContactAccess.GetAllByHospital(result.HospitalId);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHospitalGet, ex, User, LogMessage.HIPS_MESSAGE_065);
            }
            return result;
        }

        /// <summary>
        /// Gets all the hospitals, including their addresses, codes and contacts.
        /// </summary>
        /// <returns>List of hospitals</returns>
        public List<Hospital> GetAll()
        {
            return GetAll(null);
        }

        /// <summary>
        /// Gets all the hospitals, or all the hospitals that were updated after
        /// the specified date/time, including their addresses, codes and contacts.
        /// </summary>
        /// <param name="lastUpdated">The date and time of the last fetch.</param>
        /// <returns>List of hospitals</returns>
        public List<Hospital> GetAll(DateTime? lastUpdated)
        {
            List<Hospital> results = new List<Hospital>();
            try
            {
                // Populate the hospital list first
                using (SqlCommand command = GetSqlCommand("hips.HospitalGet"))
                {
                    if (lastUpdated.HasValue)
                    {
                        command.Parameters.Add(new SqlParameter("@DateModified", lastUpdated));
                    }
                    results = GetPopulatedBusinessList<Hospital>(command.ExecuteReader());
                    command.Connection.Close();
                }

                // Get all the hospital addresses, codes and contacts with one stored procedure call each
                List<Address> hospitalAddress = HospitalAddressAccess.GetAll(null);
                List<HospitalCode> hospitalCodes = HospitalCodeAccess.GetAll(null);
                List<Contact> hospitalContacts = ContactAccess.GetAllByHospital(null);

                // Assign the proper addresses, codes and contacts to each hospital
                foreach (Hospital hospital in results)
                {
                    hospital.Addresses = hospitalAddress.Where(a => a.HospitalId == hospital.HospitalId.Value).ToList();
                    hospital.Codes = hospitalCodes.Where(a => a.HospitalId == hospital.HospitalId.Value).ToList();
                    hospital.Contacts = hospitalContacts.Where(a => a.HospitalId == hospital.HospitalId.Value).ToList();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHospitalGet, ex, User, LogMessage.HIPS_MESSAGE_063);
            }
            return results;
        }

        #endregion Methods
    }
}