﻿// -----------------------------------------------------------------------
// <copyright file="CdaHelper.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Configuration;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Numerics;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// This class represents a CDA document used for CCA testing.
    /// </summary>
    public class CdaDocument
    {
        #region Private Properties

        private const string HL7_V3_NAMESPACE = "urn:hl7-org:v3";
        private const string AUSTRALIAN_CDA_EXTENSIONS_NAMESPACE = "http://ns.electronichealth.net.au/Ci/Cda/Extensions/3.0";
        private readonly string[] HL7_DATE_FORMATS = { "yyyyMMdd", "yyyyMMddHHmmss" };

        public const string CDA_IMPLEMENTATION_GUIDE_ROOT = "1.2.36.1.2001.1001.101.100.1002.4";
        public const string CDA_IMPLEMENTATION_GUIDE_EXTENSION = "3.4";

        public const string CDA_RENDERING_SPECIFICATION_ROOT = "1.2.36.1.2001.1001.100.149";
        public const string CDA_RENDERING_SPECIFICATION_EXTENSION = "1.0";


        private XmlDocument documentField;
        private XmlNamespaceManager namespaceManagerField;

        [Serializable]
        public enum CdaConformanceLevel
        {
            Unspecified = -1,
            HPII_Relaxed_1A = 1,
            HPII_Enforced_1A = 2,
        }

        private Dictionary<string, SampleDocumentType> documentTypeCodes = new Dictionary<string, SampleDocumentType>()
        {
            { "60591-5", SampleDocumentType.SharedHealthSummary },
            { "57133-1", SampleDocumentType.eReferral },
            { "51852-2", SampleDocumentType.SpecialistLetter },
            { "18842-5", SampleDocumentType.DischargeSummary3A },
            { "34133-9", SampleDocumentType.EventSummary },
            { "100.16764", SampleDocumentType.PcehrPrescriptionRecord },
            { "100.16765", SampleDocumentType.PcehrDispenseRecord },
            { "100.16957", SampleDocumentType.DiagnosticImagingReport},
            { "100.32001", SampleDocumentType.PathologyReport},
        };

        /// <summary>
        /// The document type.
        /// </summary>
        public SampleDocumentType GetDocType()
        {
            XmlNode codeNode = Document.SelectSingleNode("//x:code", NamespaceManager);
            XmlAttribute codeAttribute = codeNode.Attributes["code"];
            return documentTypeCodes[codeAttribute.Value];
        }

        /// <summary>
        /// The XmlDocument object that backs this CdaDocument object.
        /// </summary>
        private XmlDocument Document
        {
            get
            {
                if (documentField == null)
                {
                    documentField = new XmlDocument();
                }
                return documentField;
            }
        }

        /// <summary>
        /// An XmlNamespaceManager that includes the HL7 v3 (abbreviated to "x")
        /// and Australian CDA Extensions (abbreviated to "ext") namespaces.
        /// </summary>
        public XmlNamespaceManager NamespaceManager
        {
            get
            {
                if (namespaceManagerField == null)
                {
                    namespaceManagerField = new XmlNamespaceManager(new NameTable());
                    namespaceManagerField.AddNamespace("ext", AUSTRALIAN_CDA_EXTENSIONS_NAMESPACE);
                    namespaceManagerField.AddNamespace("x", HL7_V3_NAMESPACE);
                }
                return namespaceManagerField;
            }
        }

        #endregion Private Properties

        /// <summary>
        /// Loads a CDA document from the given XML file.
        /// </summary>
        /// <param name="file">File name/path information</param>
        public CdaDocument(FileInfo file)
        {
            Document.Load(file.FullName);
        }

        /// <summary>
        /// Loads a CDA document from the binary contents.
        /// </summary>
        /// <param name="cdaDocument"></param>
        public CdaDocument(byte[] cdaDocument)
        {
            using (MemoryStream stream = new MemoryStream(cdaDocument))
            {
                Document.Load(stream);
            }
        }

        /// <summary>
        /// Gets the document set identifier from a CDA document (this one stays the same between versions).
        /// </summary>
        /// <param name="cdaDocument">The CDA document.</param>
        /// <returns></returns>
        public string GetSetId()
        {
            XmlNode setIdNode = Document.SelectSingleNode("//x:setId", NamespaceManager);
            string root = setIdNode.Attributes["root"].Value;
            if (setIdNode.Attributes["extension"] == null)
            {
                return root;
            }
            else
            {
                string extension = setIdNode.Attributes["extension"].Value;
                return string.Format("{0}^{1}", root, extension);
            }
        }

        /// <summary>
        /// Gets the document instance identifier from a CDA document (this one is unique for every version of every document).
        /// </summary>
        /// <param name="cdaDocument">The CDA document.</param>
        /// <returns></returns>
        public string GetDocId()
        {
            XmlNode idNode = Document.SelectSingleNode("//x:id", NamespaceManager);
            string root = idNode.Attributes["root"].Value;
            if (idNode.Attributes["extension"] == null)
            {
                return root;
            }
            else
            {
                string extension = idNode.Attributes["extension"].Value;
                return string.Format("{0}^{1}", root, extension);
            }
        }

        public List<PatientMasterName> PatientNames
        {
            get
            {
                XmlNodeList nameNodes = Document.SelectNodes("//x:recordTarget/x:patientRole/x:patient/x:name", NamespaceManager);
                List<PatientMasterName> names = new List<PatientMasterName>();
                foreach (XmlNode node in nameNodes)
                {
                    PatientMasterName name = new PatientMasterName();
                    name.FamilyName = node.SelectSingleNode("x:family", NamespaceManager).InnerText;
                    List<string> givenNames = new List<string>();
                    foreach (XmlNode given in node.SelectNodes("x:given", NamespaceManager))
                    {
                        givenNames.Add(given.InnerText);
                    }
                    List<string> prefixes = new List<string>();
                    foreach (XmlNode prefix in node.SelectNodes("x:prefix", NamespaceManager))
                    {
                        prefixes.Add(prefix.InnerText);
                    }
                    List<string> suffixes = new List<string>();
                    foreach (XmlNode suffix in node.SelectNodes("x:suffix", NamespaceManager))
                    {
                        suffixes.Add(suffix.InnerText);
                    }
                    name.GivenNames = string.Join(" ", givenNames);
                    name.Title = string.Join(" ", prefixes);
                    name.Suffix = string.Join(" ", suffixes);
                    names.Add(name);
                }
                return names;
            }
        }

        public DateTime PatientDateOfBirth
        {
            get
            {
                XmlNode dobNode = Document.SelectSingleNode("//x:recordTarget/x:patientRole/x:patient/x:birthTime", NamespaceManager);
                return DateTime.ParseExact(dobNode.Attributes["value"].Value,
                    HL7_DATE_FORMATS,
                    CultureInfo.InvariantCulture.DateTimeFormat,
                    DateTimeStyles.None);
            }
        }

        public SexEnumerator PatientSex
        {
            get
            {
                XmlNode sexNode = Document.SelectSingleNode("//x:recordTarget/x:patientRole/x:patient/x:administrativeGenderCode", NamespaceManager);
                switch (sexNode.Attributes["code"].Value)
                {
                    case "M": return SexEnumerator.Male;
                    case "F": return SexEnumerator.Female;
                    case "I": return SexEnumerator.IntersexOrIndeterminate;
                    case "N": return SexEnumerator.NotStatedOrInadequatelyDescribed;
                }
                throw new InvalidOperationException("Unrecognised sex code in downloaded document");
            }
        }

        /// <summary>
        /// Sets the IHI to the given value. This method must allow an empty or invalid format IHI to be set because
        /// we are required to test the ability of HIPS to reject such documents and not attempt to upload them.
        /// </summary>
        /// <param name="testPatientIhi"></param>
        public void SetIhi(string testPatientIhi)
        {
            XmlNode ihiNode = Document.SelectSingleNode("//x:recordTarget/x:patientRole/x:patient/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='IHI']", NamespaceManager);
            string ihi = ihiNode.Attributes["root"].Value;
            string oidQualifier = "1.2.36.1.2001.1003.0.";
            ihiNode.Attributes["root"].Value = oidQualifier + testPatientIhi;
        }

        /// <summary>
        /// Increments the document ID. If the document ID has an extension, assumes that the root is an OID,
        /// checks that the extension is the same as the last part of the OID, and increments both before
        /// returning the new extension.
        /// 
        /// If the document ID does not have an extension, sets the root to a new UUID and returns the UUID.
        /// </summary>
        /// <returns>The new final part</returns>
        public string IncrementDocumentId()
        {
            XmlNode idNode = Document.SelectSingleNode("//x:id", NamespaceManager);
            XmlAttribute extensionAttribute = idNode.Attributes["extension"];
            string finalPart;
            if (extensionAttribute != null)
            {
                finalPart = idNode.Attributes["extension"].Value;
                string[] oidParts = idNode.Attributes["root"].Value.Split('.');
                Assert.AreEqual(finalPart, oidParts.Last());
                finalPart = oidParts[oidParts.Length - 1] = string.Format("{0}", 1 + int.Parse(finalPart));
                idNode.Attributes["root"].Value = string.Join(".", oidParts);
                idNode.Attributes["extension"].Value = finalPart;
            }
            else
            {
                finalPart = Guid.NewGuid().ToString();
                idNode.Attributes["root"].Value = finalPart;
            }
            return finalPart;
        }

        /// <summary>
        /// Saves a copy of the modified document into the given filename.
        /// </summary>
        /// <param name="filename">The file name</param>
        public void Save(string filename)
        {
            Document.Save(filename);
        }

        /// <summary>
        /// Gets the document encoded as UTF-8 without a Byte Order Mark (BOM) - this is imperative for PCEHR.
        /// </summary>
        /// <returns>The bytes of the CDA document</returns>
        public byte[] GetBytes()
        {
            MemoryStream stream = new MemoryStream();
            XmlWriter writer = XmlWriter.Create(stream, new XmlWriterSettings() { Encoding = new UTF8Encoding(false) });
            Document.Save(writer);
            return stream.ToArray();
        }

        /// <summary>
        /// Converts the UUID to an OID.
        ///
        /// Technical Service Specification for PCEHR Document Exchange
        /// Service Using the IHE XDS.b Platform, Version 1.3, 6 September 2012
        ///
        /// Where the CDA Document Identifier is represented as an UUID the
        /// XDSDocumentEntry.uniqueID field SHALL be set to an OID
        /// representation of this UUID. The OID representation SHALL be
        /// constructed as specified in section 7 of [ITU-X.667].
        ///
        /// ITU-T Recommendation X.667: Generation and registration of
        /// Universally Unique Identifiers (UUIDs) and their use as ASN.1
        /// object identifier components
        ///
        /// An OID formed using a UUID shall be:
        ///     {joint-iso-itu-t uuid(25) <uuid-single-integer-value>}
        /// where <uuid-single-integer-value> is the single integer value of
        /// the UUID specified in 6.3.
        ///
        /// A UUID can be represented as a single integer value. To obtain the
        /// single integer value of the UUID, the 16 octets of the binary
        /// representation shall be treated as an unsigned integer encoding
        /// with the most significant bit of the integer encoding as the most
        /// significant bit (bit 7) of the first of the sixteen octets (octet
        /// 15) and the least significant bit as the least significant bit (bit
        /// 0) of the last of the sixteen octets (octet 0).
        ///
        /// Ref: http://www.itu.int/ITU-T/studygroups/com17/oid/X.667-E.pdf
        /// </summary>
        /// <param name="uuid">The UUID</param>
        /// <returns>The OID that represents that UUID</returns>
        public static string ConvertToOid(string uuid)
        {
            BigInteger value = BigInteger.Parse("00" + uuid.Replace("-", ""), NumberStyles.HexNumber);
            return string.Format("2.25.{0}", value);
        }

        public XmlNode GetClinicalDocumentTitle()
        {
            XmlNode title = Document.SelectSingleNode("//x:title", NamespaceManager).FirstChild;
            return title;
        }

        public XmlNode GetLogoReference()
        {
            XmlNode logoReference = Document.SelectSingleNode("//x:observationMedia[@ID='LOGO']/x:value", NamespaceManager);
            return logoReference;
        }

        public XmlNodeList GetAllAttachmentReferences()
        {
            return Document.SelectNodes("//x:observationMedia/x:value/x:reference", NamespaceManager);
        }

        public XmlNodeList GetAllTemplateId()
        {
            return Document.SelectNodes("//x:templateId", NamespaceManager);
        }

        public XmlNode GetDocumentCode()
        {
            return Document.SelectSingleNode("//x:code", NamespaceManager);
        }

        public XmlNodeList GetAllObservationMediaValues()
        {
            return Document.SelectNodes("//x:observationMedia/x:value", NamespaceManager);
        }

        public XmlNodeList GetIdAttributes()
        {
            return Document.SelectNodes("//x:id", NamespaceManager);
        }

        private CcaTest testField;

        /// <summary>
        /// Validates the CDA document.
        /// </summary>
        /// <returns>True if the CDA document passed schema validation.</returns>
        public bool Validate(CcaTest test)
        {
            testField = test;
            Document.Schemas.Add(null, "CDA.xsd");
            Document.Schemas.Add(null, "EXTENSION.xsd");

            //Document.Schemas.Add(null, "datatypes-base-V3_0.xsd");
            //Document.Schemas.Add(null, "datatypes-V3_0.xsd");
            //Document.Schemas.Add(null, "NarrativeBlock.xsd");
            //Document.Schemas.Add(null, "POCD_MT000040-AU-V1_0.xsd");
            //Document.Schemas.Add(null, "voc-V3_0.xsd");
            schemaErrorsWereFound = false;

            try
            {
                // Validate the CDA document
                Document.Validate(new ValidationEventHandler(ValidationEventHandler));
            }
            catch (XmlSchemaValidationException ex)
            {
                testField.Log(ex.Message);
                schemaErrorsWereFound = true;
            }
            return !schemaErrorsWereFound;
        }

        private bool schemaErrorsWereFound;

        private void ValidationEventHandler(object sender, ValidationEventArgs e)
        {
            if (e.Severity == XmlSeverityType.Error)
            {
                schemaErrorsWereFound = true;
            }
            testField.Log(e.Message);
        }

        /// <summary>
        /// Gets the extension on the id element which is the author identifier,
        /// or if no extension then the root (with HPI-I prefix removed if present).
        /// </summary>
        /// <returns></returns>
        public string GetAuthorPersonEntityIdentifier()
        {
            XmlNode id = Document.SelectSingleNode("//x:author/x:assignedAuthor/x:assignedPerson/ext:asEntityIdentifier/ext:id", NamespaceManager);
            if (id == null)
            {
                return null;
            }
            XmlAttribute extension = id.Attributes["extension"];
            if (extension == null)
            {
                string hpiiPrefix = "1.2.36.1.2001.1003.0.";
                string root = id.Attributes["root"].Value;
                if (root.StartsWith(hpiiPrefix))
                {
                    root = root.Substring(hpiiPrefix.Length);
                }
                return root;
            }
            return extension.Value;
        }

        /// <summary>
        /// Gets the values for an "ext:asEntityIdentifier" element.
        /// </summary>
        /// <param name="xpath"></param>
        /// <returns></returns>
        private EntityIdentifier GetEntityIdentifier(string xpath)
        {
            EntityIdentifier ei = new EntityIdentifier();
            XmlNode eiNode = Document.SelectSingleNode(xpath, NamespaceManager);
            if (eiNode != null)
            {
                XmlNode idNode = eiNode.SelectSingleNode("ext:id", NamespaceManager);
                ei.Root = GetAttribute(idNode, "root");
                ei.Extension = GetAttribute(idNode, "extension");
                ei.AssigningAuthorityName = GetAttribute(idNode, "assigningAuthorityName");
                XmlNode nameNode = eiNode.SelectSingleNode("ext:assigningGeographicArea/ext:name", NamespaceManager);
                ei.AssigningGeographicArea = nameNode != null ? nameNode.InnerText : null;
                XmlNode codeNode = eiNode.SelectSingleNode("ext:code", NamespaceManager);
                ei.Code = GetAttribute(codeNode, "code");
                ei.CodeSystem = GetAttribute(codeNode, "codeSystem");
                ei.CodeSystemName = GetAttribute(codeNode, "codeSystemName");
            }
            return ei;
        }

        /// <summary>
        /// Gets the value of an attribute on a node. This is the same as
        /// <code>node.Attributes[name].Value</code> except that it's safe
        /// when node is null or the attribute doesn't exist; in both cases
        /// it just returns null.
        /// </summary>
        /// <param name="node">The XML node</param>
        /// <param name="name">The name of the attribute</param>
        /// <returns>The value of the attribute or null.</returns>
        private string GetAttribute(XmlNode node, string name)
        {
            if (node != null)
            {
                XmlAttribute attr = node.Attributes[name];
                if (attr != null)
                {
                    return attr.Value;
                }
            }
            return null;
        }

        /// <summary>
        /// Sets the values for an "ext:asEntityIdentifier" element, if it exists.
        /// Assumes there is an "ext:id" element. Creates or removes
        /// "ext:code" and "ext:assigningGeographicArea" as required.
        /// </summary>
        /// <param name="ei">The new entity identifier values.</param>
        /// <param name="xpath">An XPath expression that finds the "ext:asEntityIdentifier" element in the document.</param>
        private void SetEntityIdentifier(EntityIdentifier ei, string xpath)
        {
            XmlNode eiNode = Document.SelectSingleNode(xpath, NamespaceManager);
            if (eiNode != null)
            {
                XmlNode idNode = eiNode.SelectSingleNode("ext:id", NamespaceManager);
                SetAttribute(idNode, "ext", "root", ei.Root);
                SetAttribute(idNode, "ext", "extension", ei.Extension);
                SetAttribute(idNode, "ext", "assigningAuthorityName", ei.AssigningAuthorityName);
                SetAssigningGeographicAreaName(eiNode, ei.AssigningGeographicArea);
                SetCode(eiNode, ei.Code, ei.CodeSystem, ei.CodeSystemName);
            }
        }

        /// <summary>
        /// Creates and sets, or removes, the "ext:code" element on the given "ext:asEntityIdentifier" element.
        /// </summary>
        /// <param name="eiNode">The "ext:asEntityIdentifier" element.</param>
        /// <param name="code">The value for the code attribute, or null to remove the "ext:code" element.</param>
        /// <param name="codeSystem">The value for the codeSystem attribute.</param>
        /// <param name="codeSystemName">The value for the codeSystemName attribute.</param>
        private void SetCode(XmlNode eiNode, string code, string codeSystem, string codeSystemName)
        {
            XmlNode codeNode = eiNode.SelectSingleNode("ext:code", NamespaceManager);
            if (code == null)
            {
                if (codeNode != null)
                {
                    eiNode.RemoveChild(codeNode);
                }
            }
            else
            {
                if (codeNode == null)
                {
                    codeNode = documentField.CreateElement("ext", "code", AUSTRALIAN_CDA_EXTENSIONS_NAMESPACE);
                    eiNode.AppendChild(codeNode);
                }
                SetAttribute(codeNode, "ext", "code", code);
                SetAttribute(codeNode, "ext", "codeSystem", codeSystem);
                SetAttribute(codeNode, "ext", "codeSystemName", codeSystemName);
            }
        }

        /// <summary>
        /// Creates and sets, or removes, an attribute on an XML element.
        /// </summary>
        /// <param name="node">The XML element</param>
        /// <param name="prefix">The namespace prefix of the attribute</param>
        /// <param name="name">The name of the attribute</param>
        /// <param name="value">The new value for the attribute, or null to remove the attribute</param>
        private void SetAttribute(XmlNode node, string prefix, string name, string value)
        {
            XmlAttribute attr = node.Attributes[name];
            if (value == null)
            {
                if (attr != null)
                {
                    node.Attributes.Remove(attr);
                }
            }
            else
            {
                if (attr == null)
                {
                    attr = documentField.CreateAttribute(prefix, name, NamespaceManager.LookupNamespace(prefix));
                    node.Attributes.Append(attr);
                }
                attr.Value = value;
            }
        }

        /// <summary>
        /// Sets the inner text of the "ext:name" element under the
        /// "ext:assigningGeographicArea" element under the specified "ext:id" element.
        /// 
        /// Creates the "ext:assigningGeographicArea" (with attribute classCode="PLC")
        /// and "ext:name" elements if they do not exist.
        /// 
        /// If the given value is null, removes the "ext:assigningGeographicArea" node.
        /// </summary>
        /// <param name="value">The new value for the "ext:name" element.</param>
        /// <param name="id">The "ext:id" element.</param>
        private void SetAssigningGeographicAreaName(XmlNode id, string value)
        {
            XmlNode assigningGeographicAreaNode = id.SelectSingleNode("ext:assigningGeographicArea", NamespaceManager);
            if (value != null)
            {
                if (assigningGeographicAreaNode == null)
                {
                    assigningGeographicAreaNode = documentField.CreateElement("ext", "assigningGeographicArea", AUSTRALIAN_CDA_EXTENSIONS_NAMESPACE);
                    id.AppendChild(assigningGeographicAreaNode);
                }
                SetAttribute(assigningGeographicAreaNode, "ext", "classCode", "PLC");
                XmlNode nameNode = id.SelectSingleNode("ext:assigningGeographicArea/ext:name", NamespaceManager);
                if (nameNode == null)
                {
                    nameNode = documentField.CreateElement("ext", "name", AUSTRALIAN_CDA_EXTENSIONS_NAMESPACE);
                    assigningGeographicAreaNode.AppendChild(nameNode);
                }
                nameNode.InnerText = value;
            }
            else
            {
                if (assigningGeographicAreaNode != null)
                {
                    id.RemoveChild(assigningGeographicAreaNode);
                }
            }
        }

        /// <summary>
        /// Gets the extension on the id element which is the author organisation identifier,
        /// or if no extension then the root (with HPI-O prefix removed if present).
        /// </summary>
        /// <returns></returns>
        public string GetAuthorOrganisationEntityIdentifier()
        {
            XmlNode id = Document.SelectSingleNode("//x:author/x:assignedAuthor/x:assignedPerson/ext:asEmployment/ext:employerOrganization/x:asOrganizationPartOf/x:wholeOrganization/ext:asEntityIdentifier/ext:id", NamespaceManager);
            if (id == null)
            {
                return null;
            }
            XmlAttribute extension = id.Attributes["extension"];
            if (extension == null)
            {
                string hpioPrefix = "1.2.36.1.2001.1003.0.";
                string root = id.Attributes["root"].Value;
                if (root.StartsWith(hpioPrefix))
                {
                    root = root.Substring(hpioPrefix.Length);
                }
                return root;
            }
            return extension.Value;
        }

        /// <summary>
        /// Gets the family name of the document author.
        /// </summary>
        /// <returns></returns>
        public string GetAuthorFamilyName()
        {
            XmlNode family = Document.SelectSingleNode("//x:author/x:assignedAuthor/x:assignedPerson/x:name/x:family", NamespaceManager);
            if (family == null)
            {
                return null;
            }
            return family.InnerText;
        }

        /// <summary>
        /// Gets the given names of the document author.
        /// </summary>
        /// <returns></returns>
        public List<string> GetAuthorGivenNames()
        {
            XmlNodeList givens = Document.SelectNodes("//x:author/x:assignedAuthor/x:assignedPerson/x:name/x:given", NamespaceManager);
            List<string> givenNames = new List<string>();
            foreach (XmlNode given in givens)
            {
                givenNames.Add(given.InnerText);
            }
            return givenNames;
        }

        /// <summary>
        /// Sets the set ID of the document. The root must be either an OID or a UUID.
        /// </summary>
        /// <param name="root">Mandatory root part</param>
        /// <param name="extension">Optional extension part</param>
        public void SetSetId(string root, string extension = null)
        {
            XmlNode setIdNode = Document.SelectSingleNode("/x:ClinicalDocument/x:setId", NamespaceManager);
            setIdNode.Attributes["root"].Value = root;
            if (extension != null)
            {
                XmlAttribute extensionAttribute = setIdNode.Attributes["extension"];
                if (extensionAttribute == null)
                {
                    extensionAttribute = Document.CreateAttribute("extension");
                    setIdNode.Attributes.Append(extensionAttribute);
                }
                extensionAttribute.Value = extension;
            }
        }

        /// <summary>
        /// Allocate new GUIDs to each of the object IDs within the document to avoid
        /// PCEHR warnings like "The specified ID on the object to be created contains
        /// an II that is already used by an existing object".
        /// </summary>
        public void SetNewObjectIds()
        {
            // Select all id nodes in the "urn:hl7-org:v3" namespace except for the top level document ID itself.
            XmlNodeList objectIdNodes = Document.SelectNodes("/x:ClinicalDocument/*//x:id", NamespaceManager);
            Dictionary<string, string> replacements = new Dictionary<string, string>();
            foreach (XmlNode node in objectIdNodes)
            {
                XmlAttribute rootAttribute = node.Attributes["root"];
                XmlAttribute extensionAttribute = node.Attributes["extension"];
                string id = rootAttribute.Value;
                string extension = extensionAttribute != null ? extensionAttribute.Value : null;
                Guid tempGuid;
                if (Guid.TryParseExact(id, "D", out tempGuid))
                {
                    // The root is a UUID. Replace the root and extension with a new UUID.

                    if (extensionAttribute != null)
                    {
                        id = string.Format("{0}^{1}", id, extension);
                        node.Attributes.Remove(extensionAttribute);
                    }
                    if (!replacements.ContainsKey(id))
                    {
                        replacements[id] = Guid.NewGuid().ToString();
                    }
                    rootAttribute.Value = replacements[id];
                }
                else if (Guid.TryParseExact(extension, "D", out tempGuid))
                {
                    // The extension is a UUID. Replace it with a new UUID.
                    // This looks to be necessary for PCEHR Prescription Record's
                    // relatedDocument/parentDocument/id

                    if (!replacements.ContainsKey(extension))
                    {
                        replacements[extension] = Guid.NewGuid().ToString();
                    }
                    extensionAttribute.Value = replacements[extension];
                }
                else
                {
                    // Non-UUID format. Leave it alone.
                }
            }
        }

        /// <summary>
        /// Sets the date of birth of the patient in the CDA document. If the
        /// parameter is null, sets the date of birth to an empty string. Even
        /// though this is clearly invalid, attempting to upload a document
        /// with an empty DOB is a required NOC test for NPDR.
        /// </summary>
        /// <param name="dateOfBirth"></param>
        public void SetDateOfBirth(DateTime? dateOfBirth)
        {
            XmlNode dateOfBirthNode = Document.SelectSingleNode("/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/x:birthTime", NamespaceManager);
            if (dateOfBirth.HasValue)
            {
                dateOfBirthNode.Attributes["value"].Value = dateOfBirth.Value.ToString("yyyyMMdd");
            }
            else
            {
                dateOfBirthNode.Attributes["value"].Value = string.Empty;
            }
        }

        public void SetSex(SexEnumerator sex)
        {
            XmlNode sexNode = Document.SelectSingleNode("/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/x:administrativeGenderCode", NamespaceManager);
            switch (sex)
            {
                case SexEnumerator.Male:
                    sexNode.Attributes["code"].Value = "M";
                    sexNode.Attributes["displayName"].Value = "Male";
                    break;

                case SexEnumerator.Female:
                    sexNode.Attributes["code"].Value = "F";
                    sexNode.Attributes["displayName"].Value = "Female";
                    break;

                case SexEnumerator.IntersexOrIndeterminate:
                    sexNode.Attributes["code"].Value = "I";
                    sexNode.Attributes["displayName"].Value = "Intersex or indeterminate";
                    break;
            }
        }

        /// <summary>
        /// Returns the format code that is configured in app.config for this type of document.
        /// </summary>
        /// <returns>The format code (template ID).</returns>
        public string GetFormatCode()
        {
            string key = string.Format("{0}TemplateId", GetDocType());

            return ConfigurationManager.AppSettings[key].ToString();
        }

        /// <summary>
        /// Returns the version number of the document.
        /// </summary>
        /// <returns>The version number of the document.</returns>
        public int GetVersionNumber()
        {
            XmlNode versionNode = Document.SelectSingleNode("/x:ClinicalDocument/x:versionNumber", NamespaceManager);
            return int.Parse(versionNode.Attributes["value"].Value);
        }

        /// <summary>
        /// Sets the version number of the document.
        /// </summary>
        /// <param name="version">The version number.</param>
        public void SetVersion(int version)
        {
            XmlNode versionNode = Document.SelectSingleNode("/x:ClinicalDocument/x:versionNumber", NamespaceManager);
            versionNode.Attributes["value"].Value = version.ToString();
        }

        /// <summary>
        /// Adds, modifies or deletes the relatedDocument with typeCode RPLC to point to the specified parent document.
        /// </summary>
        /// <param name="parent">The parent document, or null if this should be a new document.</param>
        public void SetParentDocument(CdaDocument parent)
        {
            if (GetDocType() == SampleDocumentType.SharedHealthSummary)
            {
                // Replacement SHS must be uploaded as a new document
                parent = null;
            }

            XmlNode clinicalDocumentNode = Document.SelectSingleNode("/x:ClinicalDocument", NamespaceManager);
            XmlNode rplcNode = Document.SelectSingleNode("/x:ClinicalDocument/x:relatedDocument[@typeCode='RPLC']", NamespaceManager);

            if (parent == null)
            {
                SetSetId(Guid.NewGuid().ToString());
                SetVersion(1);
                if (rplcNode != null)
                {
                    clinicalDocumentNode.RemoveChild(rplcNode);
                }
            }
            else
            {
                SetSetId(parent.GetSetId());
                SetVersion(parent.GetVersionNumber() + 1);
                XmlNode idNode, setIdNode, versionNumberNode;
                if (rplcNode == null)
                {
                    rplcNode = Document.CreateElement("relatedDocument", HL7_V3_NAMESPACE);
                    XmlAttribute typeCodeAttribute = Document.CreateAttribute("typeCode");
                    typeCodeAttribute.Value = "RPLC";
                    rplcNode.Attributes.Append(typeCodeAttribute);
                    XmlNode componentOfNode = Document.SelectSingleNode("/x:ClinicalDocument/x:componentOf", NamespaceManager);
                    clinicalDocumentNode.InsertBefore(rplcNode, componentOfNode);
                    XmlNode parentDocumentNode = Document.CreateElement("parentDocument", HL7_V3_NAMESPACE);
                    rplcNode.AppendChild(parentDocumentNode);

                    idNode = Document.CreateElement("id", HL7_V3_NAMESPACE);
                    idNode.Attributes.Append(Document.CreateAttribute("root"));
                    parentDocumentNode.AppendChild(idNode);

                    setIdNode = Document.CreateElement("setId", HL7_V3_NAMESPACE);
                    setIdNode.Attributes.Append(Document.CreateAttribute("root"));
                    parentDocumentNode.AppendChild(setIdNode);

                    versionNumberNode = Document.CreateElement("versionNumber", HL7_V3_NAMESPACE);
                    versionNumberNode.Attributes.Append(Document.CreateAttribute("value"));
                    parentDocumentNode.AppendChild(versionNumberNode);
                }
                else
                {
                    idNode = rplcNode.SelectSingleNode("x:parentDocument/x:id", NamespaceManager);
                    setIdNode = rplcNode.SelectSingleNode("x:parentDocument/x:setId", NamespaceManager);
                    versionNumberNode = rplcNode.SelectSingleNode("x:parentDocument/x:versionNumber", NamespaceManager);
                }
                idNode.Attributes["root"].Value = parent.GetDocId();
                setIdNode.Attributes["root"].Value = parent.GetSetId();
                versionNumberNode.Attributes["value"].Value = parent.GetVersionNumber().ToString();
            }
        }

        /// <summary>
        /// This is a convenience method that calls SetEntityIdentifier with the appropriate
        /// values to change the authoring organisation, custodian, participant, and facility to the
        /// given HPI-O. It will also change the HPI-O in the OID prefix of the author person identifier,
        /// if the root starts with the HPI-O qualified local identifier OID prefix that is defined in
        /// the NEHTA FAQ.
        /// 
        /// Note for developers: update this method if you find any other places where the HPI-O appears in the document.
        /// </summary>
        /// <param name="hpio"></param>
        public void SetHpiO(string hpio)
        {
            EntityIdentifier ei = EntityIdentifier.HpiO(hpio);
            SetEntityIdentifier(ei, XPathResource.AuthorOrganisationEntityIdentifier);
            SetEntityIdentifier(ei, XPathResource.CustodianEntityIdentifier);
            SetEntityIdentifier(ei, XPathResource.ParticipantOrganisationEntityIdentifier);
            SetEntityIdentifier(ei, XPathResource.FacilityEntityIdentifier);

            EntityIdentifier authorEi = GetEntityIdentifier(XPathResource.AuthorPersonEntityIdentifier);
            if (authorEi.Root.StartsWith(EntityIdentifier.HPIO_QUALIFIED_LOCAL_IDENTIFIER_OID_PREFIX))
            {
                authorEi.Root = string.Format("{0}{1}", EntityIdentifier.HPIO_QUALIFIED_LOCAL_IDENTIFIER_OID_PREFIX, hpio);
                SetEntityIdentifier(authorEi, XPathResource.AuthorPersonEntityIdentifier);
            }
        }

        /// <summary>
        /// This is a convenience method that calls SetEntityIdentifier with the appropriate
        /// values to change the author's identifier to the given HPI-I.
        /// </summary>
        /// <param name="p"></param>
        public void SetHpiI(string hpii)
        {
            SetEntityIdentifier(EntityIdentifier.HpiI(hpii), XPathResource.AuthorPersonEntityIdentifier);
        }

        /// <summary>
        /// Sets the creation time of the CDA document. The correct format is an HL7 timestamp, such as
        /// "201401091545+1030". However this method must take a string because it's used by one of the NPDR
        /// NOC tests to deliberately set a time in the wrong format.
        /// </summary>
        /// <param name="timestamp">The new value for the document creation time.</param>
        public void SetCreationTime(string timestamp)
        {
            XmlNode effectiveTimeNode = Document.SelectSingleNode("/x:ClinicalDocument/x:effectiveTime", NamespaceManager);
            effectiveTimeNode.Attributes["value"].Value = timestamp;
        }

        /// <summary>
        /// Sets the patient name in the CDA document.
        /// </summary>
        /// <param name="familyName">The new family name</param>
        /// <param name="givenNames">The given name or names, separated by spaces</param>
        /// <param name="titles">The title or titles, separated by spaces (optional)</param>
        /// <param name="suffixes">The suffix or suffixes, separated by spaces (optional)</param>
        internal void SetPatientName(string familyName, string givenNames, string titles = null, string suffixes = null)
        {
            XmlNode nameNode = Document.SelectSingleNode("/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/x:name", NamespaceManager);
            XmlNode familyNameNode = nameNode.SelectSingleNode("x:family", NamespaceManager);
            familyNameNode.InnerText = familyName;
            ReplaceNameComponents(nameNode, "given", givenNames);
            ReplaceNameComponents(nameNode, "prefix", titles);
            ReplaceNameComponents(nameNode, "suffix", suffixes);
        }

        /// <summary>
        /// Replaces the elements for a component of a person name in a CDA document. Allowable
        /// components include "given", "prefix" or "suffix".
        /// </summary>
        /// <param name="nameNode">The "x:name" element.</param>
        /// <param name="elementName">The name of the sub-element</param>
        /// <param name="spaceSeparatedValues">The values to write into the elements, separated by spaces</param>
        private void ReplaceNameComponents(XmlNode nameNode, string elementName, string spaceSeparatedValues)
        {
            string prefix = "x";
            string xpath = string.Format("{0}:{1}", prefix, elementName);
            string namespaceUri = NamespaceManager.LookupNamespace(prefix);
            foreach (XmlNode node in nameNode.SelectNodes(xpath, NamespaceManager))
            {
                nameNode.RemoveChild(node);
            }
            if (!string.IsNullOrEmpty(spaceSeparatedValues))
            {
                foreach (string value in spaceSeparatedValues.Split(' '))
                {
                    XmlNode node = Document.CreateElement(prefix, elementName, namespaceUri);
                    node.InnerText = value;
                    nameNode.AppendChild(node);
                }
            }
        }

        /// <summary>
        /// Returns all the links whose URL scheme matches one of the given schemes.
        /// </summary>
        /// <returns>List of the target URLs for links from this document.</returns>
        public IList<string> GetAllLinks(params string[] urlSchemes)
        {
            XmlNodeList nodes = Document.SelectNodes("//x:linkHtml", NamespaceManager);
            List<string> links = new List<string>();
            foreach (XmlNode node in nodes)
            {
                Uri uri = new Uri(node.Attributes["href"].Value);
                if (urlSchemes.Contains(uri.Scheme))
                {
                    links.Add(uri.ToString());
                }
            }
            return links;
        }
    }
}