﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using HIPS.PcehrDataStore.Schemas;
using Ionic.Zip;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// This class allows multiple test methods to reuse the same uploaded document package.
    /// A shared instance is kept per document type, but reused for multiple tests.
    /// </summary>
    public class SharedPackage
    {
        public CcaPatient Patient { get; private set; }

        public CdaDocument CdaDocument { get; private set; }

        public CdaSignature CdaSignature { get; private set; }

        public ICollection<ZipEntry> PackageEntries { get; private set; }

        public ClinicalDocumentVersion Version { get; private set; }

        /// <summary>
        /// Gets the XDM-ZIP package.
        /// </summary>
        public byte[] Package { get; private set; }

        /// <summary>
        /// Gets the HL7-MDM wrapper.
        /// </summary>
        public string Wrapper { get; private set; }

        private static Dictionary<SampleDocumentType, SharedPackage> sharedInstances = new Dictionary<SampleDocumentType, SharedPackage>();

        public static SharedPackage GetSharedInstance(CcaTest test, SampleDocumentType docType)
        {
            SharedPackage sharedInstance;
            if (!sharedInstances.TryGetValue(docType, out sharedInstance))
            {
                sharedInstance = new SharedPackage();
                sharedInstance.Patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_85");
                CdaDocument cdaDocument = sharedInstance.Patient.CreateNewDocument(docType);
                
                // Old method for CDA packaging involved uploading to PCEHR and looking at the database
                QueueHelper.UploadDocumentAndWaitUntilProcessed(sharedInstance.Patient, test.LogAssert, cdaDocument);
                sharedInstance.Version = sharedInstance.Patient.GetLastClinicalDocumentVersion();
                sharedInstance.Package = sharedInstance.Version.Package;

                // New method in HIPS 5.0 will call the HIPS CDA Service
                /*
                var request = new HIPS.ServiceContracts.Cda.Message.PackageCdaRequest
                    {                        
                        Document = new HIPS.ServiceContracts.Common.DTO.CdaDocument
                        {
                            Content = cdaDocument.GetBytes(),
                        },
                        Format = HIPS.ServiceContracts.Common.CdaPackageFormat.XdmZip,
                        HospitalCode = sharedInstance.Patient.TargetPatientIdentifier.HospitalCode,
                        HospitalCodeSystem = sharedInstance.Patient.TargetPatientIdentifier.HospitalCodeSystem,
                        Recipient = new HIPS.ServiceContracts.Common.DTO.MessageAddressee
                        {
                            Organisation = new HIPS.ServiceContracts.Common.DTO.ProviderOrganisationIdentity
                            {
                                HpiO = sharedInstance.Patient.TargetHospital.HpiO,
                                Name = sharedInstance.Patient.TargetHospital.HpioName,
                            },
                        },
                        Sender = new HIPS.ServiceContracts.Common.DTO.MessageAddressee
                        {
                            Organisation = new HIPS.ServiceContracts.Common.DTO.ProviderOrganisationIdentity
                            {
                                HpiO = sharedInstance.Patient.TargetHospital.HpiO,
                                Name = sharedInstance.Patient.TargetHospital.HpioName
                            }
                        },
                        User = new HIPS.ServiceContracts.Common.DTO.UserIdentity.LocalUser
                        {
                            Login = sharedInstance.Patient.GetTestUser().Login,
                            Domain = sharedInstance.Patient.GetTestUser().Domain,
                            FamilyName = sharedInstance.Patient.GetTestUser().Name.Split(' ').Last(),
                            GivenNames = sharedInstance.Patient.GetTestUser().Name.Split(' ').First()
                        }
                    };
                var response = ProxyHelper.CdaProxy.PackageCdaDocument(request);
                sharedInstance.Package = response.Data.Content;
                request.Format = HIPS.ServiceContracts.Common.CdaPackageFormat.Hl7Mdm;
                response = ProxyHelper.CdaProxy.PackageCdaDocument(request);
                sharedInstance.Wrapper = Encoding.UTF8.GetString(response.Data.Content);
                sharedInstance.Version = new ClinicalDocumentVersion
                {
                    Package = sharedInstance.Package
                };
                */
                sharedInstance.CdaDocument = new CdaDocument(CdaPackage.ExtractDocument(sharedInstance.Package));
                sharedInstance.PackageEntries = ZipFile.Read(new MemoryStream(sharedInstance.Package)).Entries;
                sharedInstances[docType] = sharedInstance;
               
            }

            // A new test wants to use the shared instance. Reload the signature
            // because it wants a reference to the current test.
            test.patient = sharedInstance.Patient;
            sharedInstance.CdaSignature = new CdaSignature(CdaPackage.ExtractSignature(sharedInstance.Package), test);
            return sharedInstance;
        }

        public static SharedPackage GetSharedInstance1A(CcaTest test, SampleDocumentType docType)
        {
            SharedPackage sharedInstance;
            if (!sharedInstances.TryGetValue(docType, out sharedInstance))
            {
                sharedInstance = new SharedPackage();
                sharedInstance.Patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_85");
                byte[] pdf = sharedInstance.Patient.GetPDF("1A_Sample_Upload");
                QueueHelper.UploadDocument1AAndWaitUntilProcessed(sharedInstance.Patient, test.LogAssert, pdf);
                sharedInstance.Version = sharedInstance.Patient.GetLastClinicalDocumentVersion();
                sharedInstance.Package = sharedInstance.Version.Package;
                sharedInstance.CdaDocument = new CdaDocument(CdaPackage.ExtractDocument(sharedInstance.Package));
                sharedInstance.PackageEntries = ZipFile.Read(new MemoryStream(sharedInstance.Package)).Entries;
                sharedInstances[docType] = sharedInstance;
            }

            // A new test wants to use the shared instance. Reload the signature
            // because it wants a reference to the current test.
            test.patient = sharedInstance.Patient;
            sharedInstance.CdaSignature = new CdaSignature(CdaPackage.ExtractSignature(sharedInstance.Package), test);
            return sharedInstance;
        }



        /// <summary>
        /// This is a private constructor. Use GetSharedInstance method to obtain instances.
        /// </summary>
        private SharedPackage()
        {
        }
    }
}