﻿using System;
using System.Collections.Generic;
using System.Linq;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_301
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.301
    /// Use Case Name:	Access a View Service
    /// Test Case ID:   PCEHR_CIS_019041
    /// Objective:
    /// To check that the Clinical Information System provides a capability to
    /// save or print a clinical document downloaded as part of a View from the
    /// PCEHR System.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019041 : CcaTest
    {
        /// <summary>
        /// Perform an operation to request a view of the PCEHR for a
        /// healthcare recipient from the PCEHR System. If the view downloaded
        /// from the PCEHR System is a clinical document, verify that the
        /// software fulfils either criteria a or b:
        /// a. The software allows the user to save the downloaded clinical 
        ///    document to the clinical information system.
        /// b. The software allows the user to print the downloaded clinical 
        ///    document.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_301")]
        public void CIS_301_019041_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");

            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
            {
                FromDate = DateTime.Parse("1/1/2000"),
                ToDate = DateTime.Now
            };
            ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);
            CdaDocument viewDoc = new CdaDocument(viewResponse.Document);
            
            IList<string> pcehrUrls = viewDoc.GetAllLinks("pcehr");
            LogAssert.AreNotEqual(0, pcehrUrls.Count, DialogueResource.NumberOfDocumentsLinkedFromPrescriptionAndDispenseView);
            string[] parts = pcehrUrls[0].Split(':', '/');

            DocumentRequest request = new DocumentRequest();
            request.DocumentUniqueId = parts[2];
            request.RepositoryUniqueId = parts[1];
            request.SaveDocument = true;
            DocumentResponse response = ProxyHelper.PcehrProxy.RetrieveDocument(patient.TargetPatientIdentifier, patient.GetTestUser(), request);
            if (response.HipsResponse.Status == HipsResponseIndicator.DemographicMismatchWarning)
            {
                LogAssert.ExpectResponse(HipsResponseIndicator.DemographicMismatchWarning, response.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
            }
            else
            {
                LogAssert.ExpectResponse(HipsResponseIndicator.OK, response.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
            }

            // Demonstrate that HIPS has saved the downloaded clinical document
            List<DownloadedDocument> list = patient.GetAllDownloadedDocuments();
            LogAssert.AreEqual(1, list.Count, DialogueResource.DownloadedDocumentsCount);
            LogAssert.AreEqual(request.DocumentUniqueId, list[0].SourceSystemDocumentId, DialogueResource.DownloadedDocumentDocumentId);
            LogAssert.AreEqualBinary(response.Document,
                CdaPackage.ExtractDocument(list[0].Package),
                DialogueResource.DownloadedDocumentCdaXmlMatch,
                DialogueResource.DownloadedDocumentCdaXmlMismatch);
        }
    }
}