﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_501
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.501
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020204
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020204 : CcaTest
    {
        /// <summary>
        /// Prepare registration details, and complete all necessary registration form fields to result in a successful registration.
        /// a. Submit the registration request.
        /// b. Confirm that the registration process is successful.
        /// c. Verify the audit log entry contains at minimum the following information:
        ///     • Authorised Employee’s unique local system identifier (including HPI-I & full name, if available)
        ///     • Consumer’s IHI or Medicare number or DVA number or demographics that include the full name, date of birth and sex
        ///     • Date (and time if available) of the assisted registration request.

        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020204()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 01");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that HIPS audited the request
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyRegisterPatientPayload(pcehrAudit);
            
            // Check that the audit record contains a date and time
            LogAssert.IsNotNull(pcehrAudit.DateCreated, DialogueResource.AuditRecordContainsDateAndTime);
         }

        private void VerifyRegisterPatientPayload(PcehrAudit audit)
        {
            // Confirm that the payload is well formed and complies with the gainPCEHRAccess Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");

            // Check that the patient ID was provided in the request
            XmlNode ihiNumberNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHINumber element value");

            // Check that the user ID was supplied
            XmlNode userIdNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:User/c:ID", nsmgr);
            LogAssert.AreEqual(patient.GetTestUser().Login, userIdNode.InnerText, "User ID element value");

        }
    }
}
