﻿// -----------------------------------------------------------------------
// <copyright file="NOC_DoesPcehrExist.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System.IO;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentProduction
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  doesPCEHRExist
    /// </summary>
    [TestClass]
    public class NOC_DoesPcehrExist : CcaTest
    {
        /// <summary>
        /// Test No:         18
        /// Objective/Input:
        ///
        /// Ensure a verified IHI Number format for the individual who owns the
        /// PCEHR is used to perform the operation with a unique message ID.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Confirmation that the payload is well formed and complies with
        /// doesPCEHRExist Request XML Schema
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_018()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            DoesPcehrExistResponse pcehrResponse = ProxyHelper.PcehrProxy.IsPcehrAdvertised(identifier, patient.TargetPatient.DateOfBirth, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceIsPcehrAdvertised);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.DoesPcehrExist, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the doesPCEHRExist Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/PCEHRProfile/1.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/p:doesPCEHRExist", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("p"), "PCEHR_DoesPCEHRExist.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("c"), "PCEHR_CommonTypes.xsd");
            try
            {
                doc.Validate(null, payload);
                doc.Validate(null);
                LogAssert.IsTrue(true, "Validates against SOAP, DoesPCEHRExist and CommonTypes schemas", "no");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            // Check that the patient ID was provided in the request
            XmlNode ihiNumberNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHINumber element value");
        }

        /// <summary>
        /// Test No:         19
        /// Objective/Input:
        ///
        /// Verify that when an Invalid IHI Number format is used to query the PCEHR,
        /// an error message appears and the request is not transmitted to the PCEHR System.
        ///
        /// Expected Result:
        /// The request is not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that a wrong IHI Number format was used.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_019()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            DoesPcehrExistResponse response = ProxyHelper.PcehrProxy.IsPcehrAdvertised(patient.TargetPatientIdentifier,
                patient.TargetPatient.DateOfBirth, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, response.HipsResponse, DialogueResource.HipsServiceIsPcehrAdvertised);

            // Check that the access was prevented because the IHI was invalid
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.IsTrue(patient.TargetPatientMasterId != pcehrAudit.PatientMasterId,
                "The request was not transmitted to the PCEHR System",
                "The request was transmitted to the PCEHR System");
        }

        /// <summary>
        /// Test No:         20
        /// Objective/Input:
        ///
        /// To show the ability of the client system to handle a standard error
        /// (PCEHR_ERROR_0505 - Invalid HPI-O) when an invalid HPI-O is used to
        /// perform a doesPCEHRExist operation.
        /// The client system is configured with an invalid HPI-O.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System. Client
        /// system is able to handle and inform the user regarding the invalid
        /// HPI-O ( PCEHR_ERROR_0505 - Invalid HPI-O).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_020()
        {
            // Create the patient record
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

            DoesPcehrExistResponse response = ProxyHelper.PcehrProxy.IsPcehrAdvertised(
                patient.TargetPatientIdentifier, patient.TargetPatient.DateOfBirth, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, response.HipsResponse, DialogueResource.HipsServiceIsPcehrAdvertised);
            if (patient.TargetHospital.PcehrCsp)
            {
                LogAssert.AreEqual("notAuthorised", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
                LogAssert.AreEqual("PCEHR_ERROR_0004 - Authorisation denied", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
            }
            else
            {
                LogAssert.AreEqual("badParam", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
                LogAssert.AreEqual("PCEHR_ERROR_0505 - Invalid HPI-O", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
            }
            HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
        }

        /// <summary>
        /// Test No:         21
        /// Objective/Input:
        ///
        /// To show the ability of the client system to perform a doesPCEHRExist
        /// operation for an IHI which does not exist in the PCEHR System.
        ///
        /// The IHI number used must not exist in the PCEHR System.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. Client
        /// system is able to handle the response elements in the PCEHRHeader and
        /// the body where the PCEHRExists element is false and the required Access
        /// Code is null.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_021()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 37");
            DoesPcehrExistResponse response = ProxyHelper.PcehrProxy.IsPcehrAdvertised(
                patient.TargetPatientIdentifier, patient.TargetPatient.DateOfBirth, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response.HipsResponse, DialogueResource.HipsServiceIsPcehrAdvertised);

            // Check that PCEHR Exists is false
            LogAssert.AreEqual(false, response.PcehrExists, "PCEHR Exists");

            // Check that the Access Code Required is null (returned by HIPS as Unknown).
            LogAssert.AreEqual(AccessCodeRequired.Unknown, response.AccessCodeRequired, "Access Code Required");
        }
    }
}