﻿using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using ServiceSchemas = HIPS.ServiceContracts.Common.DTO;

namespace HIPS.AppServer.ServiceHost.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the CDA Document entity.
    /// </summary>
    internal class CdaProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // DTO --> Data

            this.CreateMap<ServiceSchemas.CdaHeaderMetadata, ModelSchemas.Cda.CdaHeaderMetadata>();
            this.CreateMap<ServiceSchemas.Address, ModelSchemas.Cda.Address>();
            this.CreateMap<ServiceSchemas.ParticipatingIndividual.ParticipatingProvider, ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>();
            this.CreateMap<ServiceSchemas.ElectronicCommunicationDetail, ModelSchemas.Cda.ElectronicCommunicationDetail>();

            this.CreateMap<ServiceSchemas.CdaAttachment, ModelSchemas.Cda.CdaAttachment>();

            // Data --> DTO

            this.CreateMap<ConsentSchemas.PatientParticipationStatus, HIPS.ServiceContracts.Pcehr.Message.RefreshPatientParticipationStatusResponse>()
                .ForMember(dest => dest.Data, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, HIPS.ServiceContracts.Pcehr.Message.UploadDischargeSummaryLevel1AResponse>()
                .ForMember(dest => dest.Data, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Status, opt => opt.ResolveUsing((ModelSchemas.Cda.CdaDocumentDetails src) =>
                    src.Status.Status == ModelSchemas.HipsResponseIndicator.OK
                    ? ServiceContracts.Common.ResponseStatus.OK
                    : ServiceContracts.Common.ResponseStatus.Warning))
                .ForMember(dest => dest.Messages, opt => opt.MapFrom(
                    src => new List<string>
                    {
                        src.Status.HipsErrorMessage,
                        src.Status.ResponseCodeDescription,
                        src.Status.ResponseCodeDetails
                    }.Where(a => !string.IsNullOrEmpty(a))));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, HIPS.ServiceContracts.Cda.Message.CreateDischargeSummaryLevel1AResponse>()
                .ForMember(dest => dest.Data, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Status, opt => opt.ResolveUsing((ModelSchemas.Cda.CdaDocumentDetails src) =>
                    src.Status.Status == ModelSchemas.HipsResponseIndicator.OK
                    ? ServiceContracts.Common.ResponseStatus.OK
                    : ServiceContracts.Common.ResponseStatus.Warning))
                .ForMember(dest => dest.Messages, opt => opt.MapFrom(
                    src => new List<string>
                    {
                        src.Status.HipsErrorMessage,
                        src.Status.ResponseCodeDescription,
                        src.Status.ResponseCodeDetails
                    }.Where(a => !string.IsNullOrEmpty(a))));

            this.CreateMap<string, ServiceSchemas.Message>()
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => src));

            this.CreateMap<ConsentSchemas.PatientParticipationStatus, ServiceSchemas.PatientParticipationStatus>()
                .ForMember(dest => dest.ParticipationStatus, opt => opt.MapFrom(src => src.ParticipationStatus));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, ServiceSchemas.CdaDocumentDetails>()
                .ForMember(dest => dest.Document, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, ServiceSchemas.CdaDocument>()
                .ForMember(dest => dest.Content, opt => opt.MapFrom(src => src.CdaDocumentXmlBytes));

            // Map DocumentBodyPdf to AttachedDocumentPdf
            this.CreateMap<ModelSchemas.Cda.CdaAttachment, ServiceSchemas.CdaAttachment>()
                .ForMember(dest => dest.AttachmentType, opt => opt.MapFrom(
                    src => src.AttachmentType == ModelSchemas.Cda.AttachmentType.DocumentBodyPdf
                        ? ModelSchemas.Cda.AttachmentType.AttachedDocumentPdf
                        : src.AttachmentType));

            this.CreateMap<ModelSchemas.Cda.EntityIdentifier, string>()
                .ConstructUsing((ModelSchemas.Cda.EntityIdentifier src) => string.Format("{0}^{1}", src.Root, src.Extension));
        }

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion Methods
    }
}