﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.ConsentSchemas;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;

namespace HIPS.CommonBusinessLogic.Consent
{
    /// <summary>
    /// Business logic to check whether a patient is participating in the PCEHR system.
    /// </summary>
    public class CheckPatientParticipation
    {
        /// <summary>
        /// Gets the participation status for a single hospital patient record. This involves
        /// checking both the HI Service and the PCEHR System, as well as local consent records.
        /// <list type="ordered">
        /// <item><description>
        /// If we do not have a valid IHI then the patient is not considered participating.
        /// We cannot upload a discharge summary for a patient without a valid IHI.
        /// </description></item>
        /// <item><description>
        /// If the patient has disclosed the existence of a PCEHR to a participating health provider
        /// organisation, they are taken to have consented to the upload of any discharge
        /// summary from any hospitals in that organisation.
        /// </description></item>
        /// <item><description>
        /// If the patient has an advertised PCEHR, then we assume they do want their discharge
        /// summary to be uploaded to PCEHR.</description></item>
        /// <item><description>
        /// With no advertised PCEHR and no explicit consents in the hospital, we assume the
        /// patient will not expect that hospital to upload the discharge summary.
        /// </description></item>
        /// </list>
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier (MRN, SPI, REP, PMI, VI).</param>
        /// <param name="user">The user details for IHI validation and PCEHR advertised checking.</param>
        /// <returns>Response containing error indicator and the participation status for the specified patient</returns>
        public PatientParticipationResponse GetPatientParticipationStatus(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            PatientAccess patientAccess = new PatientAccess(user);
            PatientParticipationResponse list = new PatientParticipationResponse();
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            list.Response = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (list.Response.Status != HipsResponseIndicator.OK)
            {
                return list;
            }

            // When checking participation status, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            list.Response = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (list.Response.Status != HipsResponseIndicator.OK && list.Response.Status != HipsResponseIndicator.InvalidIhi && list.Response.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                return list;
            }
            list.PatientParticipationList = new PatientParticipationStatus[]
            {
                this.GetPatientParticipationStatus(hospital, hospitalPatient, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem, user)
            };
            return list;
        }

        /// <summary>
        /// This service will obtain the participation status for all patients with changes to their participation status since the given date/time.
        /// It will only return the participation status for patients in hospitals that have a code in the given hospital code system
        /// (e.g. those which use a particular PAS or CIS).
        /// </summary>
        /// <param name="since">The date/time after which the episode must have been modified to be included</param>
        /// <param name="hospitalCodeSystem">Code that filters which hospitals are included</param>
        /// <param name="user">The user details for IHI validation and PCEHR advertised checking.</param>
        /// <returns>List of participation status for patients</returns>
        public PatientParticipationResponse GetRecentPatientParticipationStatus(DateTime since, string hospitalCodeSystem, UserDetails user)
        {
            PatientAccess dataAccessHelper = new PatientAccess(user);
            PatientParticipationResponse response = new PatientParticipationResponse();
            List<PatientParticipationStatus> statuses = new List<PatientParticipationStatus>();
            response.Response = new HipsResponse(HipsResponseIndicator.OK);

            List<HealthProviderOrganisationPatient> hpops = dataAccessHelper.HealthProviderOrganisationPatientDataAccess.GetAll(since);
            HashSet<int> patientMasterIds = new HashSet<int>();
            foreach (HealthProviderOrganisationPatient hpop in hpops)
            {
                List<HospitalPatient> hospitalPatients = dataAccessHelper.HospitalPatientDataAccess.GetAllActive(hpop.PatientMasterId.Value);
                foreach (HospitalPatient hospitalPatient in hospitalPatients)
                {
                    Hospital hospital = dataAccessHelper.Hospitals.Where(a => a.HospitalId == hospitalPatient.HospitalId).FirstOrDefault();
                    string hospitalCode = hospital.GetCode(hospitalCodeSystem);
                    if (hospitalCode != null)
                    {
                        // Include this hospital because it is relevant to the querying system
                        PatientParticipationStatus patientStatus = this.GetPatientParticipationStatus(hospital, hospitalPatient, hospitalCode, hospitalCodeSystem, user);
                        statuses.Add(patientStatus);
                    }
                }
            }
            string infoMessage = LogMessages.InfoPatientParticipation;
            string exceptionMessage = string.Format(ExceptionMessages.InfoPatientParticipation, statuses.Count, since);
            EventLogger.WriteLog(infoMessage, new Exception(exceptionMessage), user, LogMessage.HIPS_MESSAGE_113);
            response.PatientParticipationList = statuses.ToArray();
            return response;
        }

        /// <summary>
        /// This operation looks up a patient record in the current facility (for advertised status) and disclosure facility (for disclosed status),
        /// gets a validated IHI for the patient, and optionally refreshes the advertised status from the PCEHR system.
        /// </summary>
        /// <param name="patientIdentifier">The Registered Enterprise Patient</param>
        /// <param name="disclosureFacility">Disclosure Facility</param>
        /// <param name="forceRefresh">Force Refresh Option</param>
        /// <param name="user">The user details of the authorised employee for IHI and PCEHR checks</param>
        /// <returns>The PCEHR participation status for the patient</returns>
        public PatientParticipationStatus RefreshPatientParticipationStatus(PatientIdentifierBase patientIdentifier, HospitalIdentifier disclosureFacility, ForceRefreshOption forceRefresh, UserDetails user)
        {
            // Declarations
            // ValidatedIhiResponse response = new ValidatedIhiResponse();
            Hospital currentHospital = null, disclosureFacilityHospital = null;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;

            // 1. Create an instance of the “PatientParticipationStatus” class in the “HIPS.ConsentSchemas” namespace. The response will be built up inside this object.
            PatientParticipationStatus patientParticipationStatus = new PatientParticipationStatus();

            // 2. Create an instance of the “PatientAccess” business logic and invoke “GetHospital” to look up the current facility and to look up the disclosure facility if specified otherwise use the current facility for both.
            PatientAccess patientAccess = new PatientAccess(user);
            HipsResponse hospitalResponse = patientAccess.GetHospital(patientIdentifier, out currentHospital);
            if (hospitalResponse.Status == HipsResponseIndicator.DatabaseError)
            {
                throw new HipsResponseException(hospitalResponse);
            }

            if (disclosureFacility != null && !string.IsNullOrEmpty(disclosureFacility.HospitalCode))
            {
                // Look up the disclosure facility by creating a new patient
                // identifier and setting the Hospital Code and Hospital Code
                // System from the Disclosure Facility.
                PatientIdentifierBase patientIdentifierForDisclosureFacility = new PatientMasterId();
                patientIdentifierForDisclosureFacility.HospitalCode = disclosureFacility.HospitalCode;
                patientIdentifierForDisclosureFacility.HospitalCodeSystem = disclosureFacility.HospitalCodeSystem;
                patientAccess.GetHospital(patientIdentifierForDisclosureFacility, out disclosureFacilityHospital);
            }
            else
            {
                disclosureFacilityHospital = currentHospital;
            }

            // a. If either facility cannot be found, throw an “ItemNotFoundException”
            //    with the “ItemType” set to “Hospital”.
            if (currentHospital == null)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode);
            }
            if (disclosureFacilityHospital == null)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, disclosureFacility.HospitalCodeSystem, disclosureFacility.HospitalCode);
            }

            // 3. Invoke “PopulateAndValidateUser” to check that the user information is provided.
            // a. If the user is not valid, throw an “InvalidUserException”.
            if (!User.PopulateAndValidateUser(disclosureFacilityHospital, user))
            {
                throw new InvalidUserException("User is not valid");
            }

            // 4. Invoke “GetPatient” to look up the patient using the current facility.
            // a. If the patient is not found, throw an “ItemNotFoundException” with the “ItemType” set to “Patient”.
            //    If there is any other response apart from OK or InvalidIhi, throw a “HipsResponseException”. 
            //    This would include cases like DatabaseError, UnresolvedIhiAlert, IncorrectMrn or IncorrectStatePatientId.
            HipsResponse patientAccessResponse = patientAccess.GetPatient(patientIdentifier, currentHospital, out hospitalPatient, out patientMaster);
            if (patientAccessResponse.Status == HipsResponseIndicator.InvalidPatient)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Patient, currentHospital.Description, patientIdentifier.ToString());
            }
            if (patientAccessResponse.Status != HipsResponseIndicator.OK && patientAccessResponse.Status != HipsResponseIndicator.InvalidIhi)
            {
                throw new HipsResponseException(patientAccessResponse);
            }
            
            if (patientMaster == null)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Patient, currentHospital.Description, patientIdentifier.ToString());
            }

            // Store the hospital code and patient’s MRN and State Patient ID into the response.
            patientParticipationStatus.HospitalCode = patientIdentifier.HospitalCode;
            patientParticipationStatus.Mrn = hospitalPatient.Mrn;
            patientParticipationStatus.StatePatientId = patientMaster.StatePatientId;

            // 5. Use the “Get” method of the “HealthProviderOrganisationPatientDataAccess” object to look up the disclosure status of the patient, passing the HPI-O of the disclosure facility.
            HealthProviderOrganisationPatient hpioPatientForDisclosureFacility;
            patientAccess.HealthProviderOrganisationPatientDataAccess.Get(disclosureFacilityHospital.HpiO, (int)patientMaster.PatientMasterId, out hpioPatientForDisclosureFacility);

            // Advertised should be checked against the currentFacility. This will get the HPIO patient based on the Current Facility to get the Advertised Property
            HealthProviderOrganisationPatient hpioPatientForCurrentFacility;
            patientAccess.HealthProviderOrganisationPatientDataAccess.Get(currentHospital.HpiO, (int)patientMaster.PatientMasterId, out hpioPatientForCurrentFacility);

            // 6. Create an instance of the “PatientIhiValidation” business logic and invoke “GetValidatedIhi”. This will validate the IHI if it is stale.
            PatientIhiValidation patientIhiValidation = new PatientIhiValidation();
            IhiSearchResponse ihiSearchResponse = patientIhiValidation.GetValidatedIhi(patientIdentifier, disclosureFacilityHospital, user, patientMaster);

            // a. If the response indicator is “HiServiceError”, throw a “HiServiceException”.
            if (ihiSearchResponse.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
            {
                throw new HiServiceException(ihiSearchResponse.HipsResponse);
            }

            // b. If the response error is “PcehrServiceError”, throw a “PcehrServiceException”.
            if (ihiSearchResponse.HipsResponse.Status == HipsResponseIndicator.PcehrServiceError)
            {
                throw new PcehrServiceException(ihiSearchResponse.HipsResponse);
            }

            // c. Otherwise if the response error is not “OK”, throw a “HipsResponseException”.
            if (ihiSearchResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                throw new HipsResponseException(ihiSearchResponse.HipsResponse);
            }

            // 7. Invoke “ValidateLocalIhiInformation” to check that the patient has a valid IHI.
            // a. If the response indicator is not “OK”, set ParticipationStatus.NoValidIhi
            //    and exit the function with only the data that we have gathered.
            HipsResponse localIhiStatus = new HipsResponse(HipsResponseIndicator.OK);
            patientAccess.ValidateLocalIhiInformation(patientMaster, localIhiStatus);

            if (localIhiStatus.Status != HipsResponseIndicator.OK)
            {
                patientParticipationStatus.ParticipationStatus = ParticipationStatus.NoValidIhi;
                return patientParticipationStatus;
            }

            // 8. Store the validated IHI information and the advertised and disclosed status into the response.
            patientParticipationStatus.ValidatedIhi = new ValidatedIhi(
                patientMaster.Ihi,
                (IhiStatus)patientMaster.IhiStatusId,
                (IhiRecordStatus)patientMaster.IhiRecordStatusId,
                patientMaster.IhiLastValidated.Value,
                patientMaster.RegisteredFamilyName,
                patientMaster.RegisteredGivenName,
                patientMaster.RegisteredDateOfBirth,
                (SexEnumerator)patientMaster.RegisteredSexId,
                patientIdentifier.HospitalCode,
                patientIdentifier.HospitalCodeSystem);

            // Populate the Participation Status using the advertised flag
            // from the current facility and the disclosed flag from the
            // disclosure facility.
            patientParticipationStatus.ParticipationStatus = this.GetParticipationStatus(
                hpioPatientForCurrentFacility.PcehrAdvertised, 
                hpioPatientForDisclosureFacility.PcehrDisclosed);

            // 9. If the “ForceRefresh” parameter is set to “Never”, return the patient participation status without refreshing the PCEHR status.
            if (forceRefresh == ForceRefreshOption.Never)
            {
                return patientParticipationStatus;
            }

            // 10. If the “ForceRefresh” parameter is set to “WhenNotAdvertised”,
            //     and either advertised or disclosed is true, return the patient
            //     participation status without refreshing the PCEHR status.
            if (forceRefresh == ForceRefreshOption.WhenNotAdvertised 
                && (hpioPatientForDisclosureFacility.PcehrDisclosed
                || hpioPatientForCurrentFacility.PcehrAdvertised == true))
            {
                return patientParticipationStatus;
            }

            // Throw an Exception if IHI is null
            if (string.IsNullOrEmpty(patientMaster.Ihi))
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Patient, currentHospital.Description, patientIdentifier.ToString());
            }

            // 11. Create an instance of the “DoesPcehrExist” business logic and invoke
            //     “PcehrExists”. This will refresh the PCEHR advertised status of the patient.
            //  a. If the response indicator is not “OK”, throw a “PcehrServiceException” 
            //     containing details from the “HipsResponse” object.
            DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
            DoesPcehrExistResponse doesPcehrExistResponse = doesPcehrExist.PcehrExists(patientIdentifier, currentHospital, patientMaster, user);
            if (doesPcehrExistResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                throw new PcehrServiceException(doesPcehrExistResponse.HipsResponse);
            }
            else
            {
                // Populate the ParticipationStatus
                patientParticipationStatus.ParticipationStatus = this.GetParticipationStatus(
                    doesPcehrExistResponse.PcehrExists, 
                    hpioPatientForDisclosureFacility.PcehrDisclosed);
            }

            // 12. Return the response including the new advertised status.
            return patientParticipationStatus;
        }

        /// <summary>
        /// Get the participation status based on the value of the PCEHR advertised and PCEHR disclosed flags.
        /// </summary>
        /// <param name="pcehrAdvertised">Whether the existence of the PCEHR record is advertised to the HPO.</param>
        /// <param name="pcehrDisclose">Whether the patient has disclosed the existence of the PCEHR record to the HPO.</param>
        /// <returns>The patient's participation status.</returns>
        private ParticipationStatus GetParticipationStatus(bool? pcehrAdvertised, bool pcehrDisclose)
        {
            if (pcehrDisclose)
            {
                return ParticipationStatus.RequestedUpload;
            }
            else
            {
                bool isPcehrAdvertised = pcehrAdvertised.HasValue && pcehrAdvertised.Value;
                if (isPcehrAdvertised)
                {
                    return ParticipationStatus.PcehrAdvertised;
                }
                else
                {
                    return ParticipationStatus.PcehrNotAdvertised;
                }
            }
        }

        /// <summary>
        /// Gets the participation status of a single hospital patient record.
        /// If we do not have a valid IHI then the patient cannot be participating.
        /// If the patient has explicitly consented to the upload of a discharge summary then they are considered participating.
        /// If the patient has an advertised PCEHR, then we assume they do want to participate.
        /// With no advertised PCEHR and no explicit consents, we assume they are not participating.
        /// </summary>
        /// <param name="hospital">The healthcare facility that is requesting the participation status.</param>
        /// <param name="hospitalPatient">The hospital patient record.</param>
        /// <param name="hospitalCode">Code identifying the hospital (HospitalCode.Code)</param>
        /// <param name="hospitalCodeSystem">Code identifying the type of hospital code (CodeSystem.Code)</param>
        /// <param name="user">The user details of the authorised employee for IHI and PCEHR checks</param>
        /// <returns>The PCEHR participation status for the patient</returns>
        private PatientParticipationStatus GetPatientParticipationStatus(Hospital hospital, HospitalPatient hospitalPatient, string hospitalCode, string hospitalCodeSystem, UserDetails user)
        {
            PatientAccess dataAccessHelper = new PatientAccess(user);
            PatientMaster patientMaster;
            if (HipsResponseIndicator.OK != dataAccessHelper.PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, hospital.HealthProviderOrganisationNetworkId, out patientMaster).Status)
            {
                return null;
            }

            // Create a patient participation status record
            PatientParticipationStatus patientStatus = new PatientParticipationStatus();
            patientStatus.Mrn = hospitalPatient.Mrn;
            patientStatus.StatePatientId = patientMaster.StatePatientId;
            patientStatus.ValidatedIhi = this.GetValidatedIhi(patientMaster, hospitalCode, hospitalCodeSystem);
            patientStatus.HospitalCode = hospitalCode;
            HealthProviderOrganisationPatient hpioPatient;
            dataAccessHelper.HealthProviderOrganisationPatientDataAccess.Get(hospital.HpiO, hospitalPatient.PatientMasterId, out hpioPatient);

            HipsResponse localIhiStatus = new HipsResponse(HipsResponseIndicator.OK);
            dataAccessHelper.ValidateLocalIhiInformation(patientMaster, localIhiStatus);

            if (localIhiStatus.Status != HipsResponseIndicator.OK)
            {
                patientStatus.ParticipationStatus = ParticipationStatus.NoValidIhi;
            }
            else if (hpioPatient.PcehrDisclosed)
            {
                patientStatus.ParticipationStatus = ParticipationStatus.RequestedUpload;
            }
            else
            {
                if (hpioPatient.PcehrAdvertised.HasValue && hpioPatient.PcehrAdvertised.Value)
                {
                    patientStatus.ParticipationStatus = ParticipationStatus.PcehrAdvertised;
                }
                else
                {
                    patientStatus.ParticipationStatus = ParticipationStatus.PcehrNotAdvertised;
                }
            }
            return patientStatus;
        }

        /// <summary>
        /// If the patient master has a validated IHI, returns an object that encapsulates the validated IHI information, otherwise null.
        /// </summary>
        /// <param name="patientMaster">The patient master record.</param>
        /// <param name="hospitalCode">Code identifying the hospital (HospitalCode.Code)</param>
        /// <param name="hospitalCodeSystem">Code identifying the type of hospital code (CodeSystem.Code)</param>
        /// <returns>an object that encapsulates the validated IHI information</returns>
        private ValidatedIhi GetValidatedIhi(PatientMaster patientMaster, string hospitalCode, string hospitalCodeSystem)
        {
            if (patientMaster.IhiLastValidated.HasValue && !string.IsNullOrEmpty(patientMaster.Ihi))
            {
                ValidatedIhi ihi = new ValidatedIhi(
                    patientMaster.Ihi,
                    (IhiStatus)patientMaster.IhiStatusId,
                    (IhiRecordStatus)patientMaster.IhiRecordStatusId,
                    patientMaster.IhiLastValidated.Value,
                    patientMaster.RegisteredFamilyName,
                    patientMaster.RegisteredGivenName,
                    patientMaster.RegisteredDateOfBirth,
                    (SexEnumerator)patientMaster.RegisteredSexId,
                    hospitalCode,
                    hospitalCodeSystem);
                return ihi;
            }
            else
            {
                // IHI has not been validated
                return null;
            }
        }
    }
}