﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;

namespace HIPS.CommonBusinessLogic
{
    public class PatientAccess : BaseDl
    {
        #region Private Fields

        /// <summary>
        /// The episode type code that is used for ValidatedIhi stub episodes.
        /// </summary>
        private const string EPISODE_STUB_CODE = "STUB";

        /// <summary>
        /// The tolerance for how closely the admission date/time must be specified
        /// in order to match to an existing episode record. Currently set to 1.0
        /// minutes.
        /// </summary>
        private const double MAXIMUM_ADMISSION_TIME_DIFFERENCE_MINUTES = 1.0;

        private ClinicalDocumentDl documentDataAccess;
        private EpisodeDl episodeDataAccess;
        private HealthProviderOrganisationPatientDl healthProviderOrganisationPatientDataAccess;
        private HospitalDl hospitalDataAccess;
        private List<Hospital> hospitals;
        private PatientMasterIhiDl ihiDataAccess;
        private PatientMasterDl masterDataAccess;
        private HospitalPatientDl patientDataAccess;
        private PendingIhiPcehrLookupDl pendingIhiPcehrLookupDataAccess;
        private ClinicalDocumentVersionDl versionDataAccess;

        #endregion Private Fields

        #region Constructors

        public PatientAccess(UserDetails user)
        {
            base.User = user;
        }

        #endregion Constructors

        #region Properties

        public ClinicalDocumentDl DocumentDataAccess
        {
            get
            {
                if (documentDataAccess == null)
                {
                    documentDataAccess = new ClinicalDocumentDl(User);
                }
                return documentDataAccess;
            }
        }

        public EpisodeDl EpisodeDataAccess
        {
            get
            {
                if (episodeDataAccess == null)
                {
                    episodeDataAccess = new EpisodeDl(User);
                }
                return episodeDataAccess;
            }
        }

        public HealthProviderOrganisationPatientDl HealthProviderOrganisationPatientDataAccess
        {
            get
            {
                if (healthProviderOrganisationPatientDataAccess == null)
                {
                    healthProviderOrganisationPatientDataAccess = new HealthProviderOrganisationPatientDl(User);
                }
                return healthProviderOrganisationPatientDataAccess;
            }
        }

        public HospitalDl HospitalDataAccess
        {
            get
            {
                if (hospitalDataAccess == null)
                {
                    hospitalDataAccess = new HospitalDl();
                }
                return hospitalDataAccess;
            }
        }

        public HospitalPatientDl HospitalPatientDataAccess
        {
            get
            {
                if (patientDataAccess == null)
                {
                    patientDataAccess = new HospitalPatientDl(User);
                }
                return patientDataAccess;
            }
        }

        public List<Hospital> Hospitals
        {
            get
            {
                if (hospitals == null)
                {
                    hospitals = HospitalDataAccess.GetAll();
                }
                return hospitals;
            }
        }

        public PatientMasterIhiDl IhiDataAccess
        {
            get
            {
                if (ihiDataAccess == null)
                {
                    ihiDataAccess = new PatientMasterIhiDl(User);
                }
                return ihiDataAccess;
            }
        }

        public PatientMasterDl PatientMasterDataAccess
        {
            get
            {
                if (masterDataAccess == null)
                {
                    masterDataAccess = new PatientMasterDl(User);
                }
                return masterDataAccess;
            }
        }

        public PendingIhiPcehrLookupDl PendingIhiPcehrLookupDataAccess
        {
            get
            {
                if (pendingIhiPcehrLookupDataAccess == null)
                {
                    pendingIhiPcehrLookupDataAccess = new PendingIhiPcehrLookupDl(User);
                }
                return pendingIhiPcehrLookupDataAccess;
            }
        }

        public ClinicalDocumentVersionDl VersionDataAccess
        {
            get
            {
                if (versionDataAccess == null)
                {
                    versionDataAccess = new ClinicalDocumentVersionDl(User);
                }
                return versionDataAccess;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Attempts to find the matching episode given an source system episode id.
        /// If the hospital patient has more than one episode with the given id then none match.
        /// </summary>
        /// <param name="sourceSystemEpisodeId">The source system episode identifier.</param>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <returns>
        /// The matched episode, or null if none matched
        /// </returns>
        public Episode GetEpisode(string sourceSystemEpisodeId, HospitalPatient hospitalPatient)
        {
            //Specific Episode must be found from the SourceSystemEpisode (Visit Number) in this case, and so all episodes
            //can be returned, including cancelled, as the result will be filtered specifically by the sourceSystemEpisodeId.
            var matched = from episode in EpisodeDataAccess.GetAll(hospitalPatient.PatientId, null)
                          where episode.SourceSystemEpisodeId == sourceSystemEpisodeId
                          select episode;
            if (matched.Count() == 1)
            {
                return matched.First();
            }
            return null;
        }

        /// <summary>
        /// Attempts to find the matching episode given an source system episode id and the Patient Id.
        /// If the hospital patient has more than one episode with the given source system episode id then none match
        /// and and error will be returned.
        /// This will also return
        /// </summary>
        /// <param name="admissionDate">The admission date/time</param>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <returns>The matched episode, or null if none matched</returns>
        public EpisodePatientExtendedDetails GetEpisodePatientExtendedDetails(string sourceSystemEpisodeId, HospitalPatient hospitalPatient, PatientIdentifierBase patientIdentifier)
        {
            //Specific Episode must be found from the SourceSystemEpisode (Visit Number) in this case, and so all episodes
            //can be returned, including cancelled, as the result will be filtered specifically by the sourceSystemEpisodeId.
            EpisodePatientExtendedDetails episode = EpisodeDataAccess.GetEpisodePatientExtendedDetails(sourceSystemEpisodeId, hospitalPatient, patientIdentifier.HospitalCodeSystem);
            return episode;
        }

        /// <summary>
        /// Attempts to find (or create) the matching episode given a document
        /// set ID and/or admission date/time.
        ///
        /// If a set ID is provided and a stub episode has been created for this
        /// set ID, then that episode is returned. In addition, if its admission
        /// date/time is outside the configured tolerance from the provided
        /// admission date/time, then the stub's admission date/time is changed
        /// to the provided one.
        ///
        /// If the hospital patient has more than one episode within the
        /// configured tolerance then none match. The episodes will need to be
        /// merged, either on the hospital PAS if supported, or by manual
        /// submission of an A35 message to HIPS, or by direct database
        /// manipulation.  Any Cancelled episodes will not be returned
        ///
        /// If the patient identifier is a validated IHI and the episode is not
        /// found through either of the mechanisms described above, then this
        /// method will create and insert a stub record for the episode. If a
        /// document set ID was provided (e.g. for a document upload operation)
        /// then the stub will be identified by the document set ID. Otherwise,
        /// there is no set ID (e.g. for a consent withdrawal), and so the stub
        /// will be identified by the admission date/time.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier</param>
        /// <param name="admissionDate">The admission date/time</param>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <param name="sourceSystemSetId">The set ID of a previously uploaded document (optional)</param>
        /// <returns>The matched episode, or null if none matched</returns>
        /// <exception cref="System.Exception">If a database error occurs while updating or inserting a stub episode</exception>
        public Episode GetEpisodeWithoutCancelled(PatientIdentifierBase patientIdentifier, DateTime admissionDate, HospitalPatient hospitalPatient, string sourceSystemSetId = null)
        {
            List<Episode> episodes = EpisodeDataAccess.GetAll(hospitalPatient.PatientId, null, true);
            EpisodeType stubType = ListSingleton.Instance.AllEpisodeTypes.FirstOrDefault(a => a.Code == EPISODE_STUB_CODE);
            Episode stub = episodes.FirstOrDefault(a => a.SourceSystemEpisodeId == sourceSystemSetId && a.EpisodeTypeId == stubType.EpisodeTypeId.Value);
            if (stub != null)
            {
                if (Math.Abs((stub.AdmissionDate - admissionDate).TotalMinutes) > MAXIMUM_ADMISSION_TIME_DIFFERENCE_MINUTES)
                {
                    stub.AdmissionDate = admissionDate;
                    if (!EpisodeDataAccess.Update(stub, null))
                    {
                        throw new Exception(string.Format(ResponseStrings.DatabaseError, EpisodeDataAccess.GetType().FullName));
                    }
                }
                return stub;
            }
            var matched = from ep in episodes
                          where Math.Abs((ep.AdmissionDate - admissionDate).TotalMinutes) < MAXIMUM_ADMISSION_TIME_DIFFERENCE_MINUTES
                          select ep;
            if (matched.Count() == 1)
            {
                return matched.First();
            }
            if (matched.Count() == 0 && patientIdentifier is ValidatedIhi)
            {
                stub = new Episode();
                stub.PatientId = hospitalPatient.PatientId.Value;
                stub.AdmissionDate = admissionDate;
                stub.SourceSystemEpisodeId = sourceSystemSetId ?? admissionDate.ToString("yyyyMMddHHmmss");
                stub.ResponsibleProviderId = -1;
                stub.EpisodeLifecycleId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.EpisodeLifecycle.Admitted;
                stub.EpisodeTypeId = stubType.EpisodeTypeId.Value;
                if (!EpisodeDataAccess.Insert(stub, null))
                {
                    throw new Exception(string.Format(ResponseStrings.DatabaseError, EpisodeDataAccess.GetType().FullName));
                }
                return stub;
            }
            return null;
        }

        /// <summary>
        /// Finds the hospital using either the internal hospital ID (if the supplied patient identifier is of PatientMasterId type) or the hospital code and hospital code system.
        /// </summary>
        /// <param name="patientIdentifier">The supplied patient identifier</param>
        /// <param name="hospital">The hospital that was found</param>
        /// <returns>Success or Failure indicator</returns>
        public HipsResponse GetHospital(PatientIdentifierBase patientIdentifier, out Hospital hospital)
        {
            if (HospitalSingleton.Value.IsHospitalListEmpty())
            {
                // Try to recover from database being down on HIPS startup.
                ListSingleton.Instance.Refresh(force: true);
            }

            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            PatientMasterId patientMasterId = patientIdentifier as PatientMasterId;
            if (patientMasterId != null && patientMasterId.HospitalId.HasValue)
            {
                hospital = HospitalSingleton.Value.Find(patientMasterId.HospitalId.Value);
            }
            else
            {
                hospital = HospitalSingleton.Value.Find(patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            }

            if (hospital == null)
            {
                if (HospitalSingleton.Value.IsHospitalListEmpty())
                {
                    response.Status = HipsResponseIndicator.DatabaseError;
                    response.HipsErrorMessage = ResponseStrings.UnableToGetHospitalList;
                }
                else
                {
                    response.Status = HipsResponseIndicator.InvalidHospital;
                    response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                }
                return response;
            }
            return response;
        }

        /// <summary>
        /// Based on the type of patient identifier supplied, reads the hospital patient and patient master from the database.
        /// In case of the type Validated IHI, it will automatically create minimal records with a dummy MRN for those
        /// jurisdictions who plan to avoid using HIPS as an HI Service platform.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">Returns the hospital patient.</param>
        /// <param name="patientMaster">Returns the hospital patient.</param>
        /// <returns>Success (OK) or reason for failure (DatabaseError or InvalidPatient or InvalidIhi or UnresolvedIhiAlert or SystemError)</returns>
        public HipsResponse GetPatient(PatientIdentifierBase patientIdentifier, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster, bool allowDemographics = false)
        {
            HipsResponse response;
            if (patientIdentifier is Mrn)
            {
                response = GetPatient((patientIdentifier as Mrn).Value, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is ValidatedIhi)
            {
                response = ReadCreateOrUpdatePatient(patientIdentifier as ValidatedIhi, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is StatePatientId)
            {
                response = GetPatient(patientIdentifier as StatePatientId, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is PatientMasterId)
            {
                response = GetPatient(patientIdentifier as PatientMasterId, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is RegisteredEnterprisePatient)
            {
                response = GetPatient(patientIdentifier as RegisteredEnterprisePatient, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is Demographic)
            {
                if (allowDemographics)
                {
                    response = GetPatient(patientIdentifier as Demographic, hospital, out hospitalPatient, out patientMaster);
                }
                else
                {
                    patientMaster = null;
                    hospitalPatient = null;
                    response = new HipsResponse(HipsResponseIndicator.SystemError, ResponseStrings.DemographicsNotAllowedForThisService);
                }
            }
            else
            {
                patientMaster = null;
                hospitalPatient = null;
                response = new HipsResponse(HipsResponseIndicator.SystemError, ResponseStrings.UnknownPatientIdentifierType);
            }
            if (response.Status == HipsResponseIndicator.OK)
            {
                ValidateLocalIhiInformation(patientMaster, response);
            }
            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given patient master ID.
        /// </summary>
        /// <param name="patientMasterId">The patient master ID</param>
        /// <param name="hospital">The hospital</param>
        /// <param name="hospitalPatient">Returns the hospital patient.</param>
        /// <param name="patientMaster">Returns the patient master.</param>
        /// <returns>Success (OK) or reason for failure (DatabaseError or InvalidPatient)</returns>
        public HipsResponse GetPatient(PatientMasterId patientMasterId, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = PatientMasterDataAccess.Get(patientMasterId.Value, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            response = HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given state patient identifier.
        /// </summary>
        /// <param name="mrn"></param>
        /// <param name="hospital"></param>
        /// <param name="hospitalPatient"></param>
        /// <param name="patientMaster"></param>
        /// <returns>Information about success or failure.</returns>
        public HipsResponse GetPatient(StatePatientId statePatientId, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = PatientMasterDataAccess.GetByStatePatientId(statePatientId.Value, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                patientMaster = null;
                hospitalPatient = null;
                return response;
            }

            response = HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given MRN.
        /// </summary>
        /// <param name="mrn"></param>
        /// <param name="hospital"></param>
        /// <param name="hospitalPatient"></param>
        /// <param name="patientMaster"></param>
        /// <returns></returns>
        public HipsResponse GetPatient(string mrn, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = HospitalPatientDataAccess.Get(hospital.HospitalId.Value, mrn, out hospitalPatient);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            response = PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given demographics.
        /// </summary>
        /// <param name="demographic">The demographics.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <returns></returns>
        public HipsResponse GetPatient(Demographic demographic, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.SystemError);
            this.LookupByDemographicsWithinHospital(demographic.FamilyName, demographic.GivenName, demographic.DateOfBirth, (int)demographic.Sex, demographic.MedicareNumber, demographic.MedicareIrn, demographic.DvaNumber, hospital, out patientMaster);
            if (patientMaster == null)
            {
                response.Status = HipsResponseIndicator.SystemError;
                hospitalPatient = new HospitalPatient();
                return response;
            }

            if (!patientMaster.PatientMasterId.HasValue)
            {
                // no matching patient so add them
                this.CreatePatient(demographic, hospital, out hospitalPatient, out patientMaster);
                response.Status = HipsResponseIndicator.OK;
            }
            else
            {
                response = HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    response.Status = HipsResponseIndicator.SystemError;
                    return response;
                }
            }

            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given Registered Enterprise Patient.
        /// </summary>
        /// <param name="registeredEnterprisePatient"></param>
        /// <param name="hospital"></param>
        /// <param name="hospitalPatient"></param>
        /// <param name="patientMaster"></param>
        /// <returns></returns>
        public HipsResponse GetPatient(RegisteredEnterprisePatient registeredEnterprisePatient, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response;
            hospitalPatient = null;
            patientMaster = null;

            // 1. Use the “GetByStatePatientId” method of the “PatientMasterDataAccess” object to look up the patient master.
            // a. If the response is not “OK”, the patient does not exist or there was a database error. Set the output parameters to null and return the response.
            response = PatientMasterDataAccess.GetByStatePatientId(registeredEnterprisePatient.StatePatientId, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;

                return response;
            }

            // 2. Use the “Get” method of the “HospitalPatientDataAccess” object to look up the existing hospital patient with the specified MRN.
            response = HospitalPatientDataAccess.Get(hospital.HospitalId.Value, registeredEnterprisePatient.Mrn, out hospitalPatient);

            // a. If the response is “OK” and the hospital patient’s PatientMasterId is the same as the patient master’s PatientMasterId, the lookup is successful so return the response.
            // b. If the response is “OK” and the hospital patient’s PatientMasterId is different to the patient master’s PatientMasterId, it means the hospital patient is on the wrong patient master.
            if (response.Status == HipsResponseIndicator.OK)
            {
                if (patientMaster.PatientMasterId == hospitalPatient.PatientMasterId)
                {
                    return response;
                }
                else
                {
                    // i. Format an error message “Patient with MRN {0} is registered in hospital {1}, but has ID {2} different to the given ID {3}”, using the MRN, hospital description, and the two StatePatientIds as parameters.
                    // ii. Write the error message to the system error log.
                    // iii. Set the response’s Status to “IncorrectStatePatientId” and set the HipsErrorMessage to the error message.
                    PatientMaster wrongPatientMaster;
                    PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, hospital.HealthProviderOrganisationNetworkId, out wrongPatientMaster);
                    string errorMessage = string.Format(EventLoggerStrings.RegisteredEnterprisePatientHasMultipleIds, registeredEnterprisePatient.Mrn, hospital.Description, wrongPatientMaster.StatePatientId, registeredEnterprisePatient.StatePatientId);
                    EventLogger.WriteLog(errorMessage, null, User, LogMessage.HIPS_MESSAGE_172);
                    response.Status = HipsResponseIndicator.IncorrectStatePatientId;
                    response.HipsErrorMessage = errorMessage;
                }
            }
            else
            {
                // 3. If the response is not “OK”, the specified MRN doesn’t yet exist. Use the “GetActive” method of the “HospitalPatientDataAccess” object to look up the current MRN of the patient in the hospital.
                //    a. If the response was “OK”, it means the patient is currently registered using a different MRN in this hospital.
                //      i. Format an error message “Patient with ID {0} is already registered in hospital {1}, but has an MRN {2} different to the given MRN {3}”, using the StatePatientId, hospital description and the two MRNs as parameters.
                //      ii. Write the error message to the system error log.
                //      iii. Set the response’s Status to “IncorrectMrn” and set the HipsErrorMessage to the error message.

                response = HospitalPatientDataAccess.GetActive((int)hospital.HospitalId.Value, (int)patientMaster.PatientMasterId, out hospitalPatient);
                if (response.Status == HipsResponseIndicator.OK)
                {
                    string errorMessage = string.Format(EventLoggerStrings.RegisteredEnterprisePatientAlreadyResigiteredAtHospital, registeredEnterprisePatient.StatePatientId, hospital.Description, hospitalPatient.Mrn, registeredEnterprisePatient.Mrn);
                    EventLogger.WriteLog(errorMessage, null, User, LogMessage.HIPS_MESSAGE_173);
                    response.Status = HipsResponseIndicator.IncorrectMrn;
                    response.HipsErrorMessage = errorMessage;
                }
                else if (response.Status == HipsResponseIndicator.InvalidPatient)
                {
                    // b. If the response was “InvalidPatient”, this means we can create the HospitalPatient object:
                    //  i. Create a new “HospitalPatient” object.
                    //  ii. Set the PatientMasterId from the “patientMaster" that was found.
                    //  iii. Set the HospitalId from the “hospital” parameter.
                    //  iv. Set the Mrn from the “patientIdentifier” parameter.
                    //  v. Invoke the “Insert” method of the “HospitalPatientDataAccess” object.
                    //  vi. If the insert fails, set the response Status to “DatabaseError”, set the response HipsErrorMessage to “Database access failed when creating patient record.” and write this message to the system error log.

                    hospitalPatient = new HospitalPatient();
                    hospitalPatient.PatientMasterId = (int)patientMaster.PatientMasterId;
                    hospitalPatient.HospitalId = (int)hospital.HospitalId.Value;
                    hospitalPatient.Mrn = registeredEnterprisePatient.Mrn;

                    bool isSuccess = HospitalPatientDataAccess.Insert(hospitalPatient);
                    if (!isSuccess)
                    {
                        response.Status = HipsResponseIndicator.DatabaseError;
                        response.HipsErrorMessage = EventLoggerStrings.RegisteredEnterprisePatientDatabaseFailure;
                        EventLogger.WriteLog(EventLoggerStrings.RegisteredEnterprisePatientDatabaseFailure, null, User, LogMessage.HIPS_MESSAGE_174);
                    }
                    else
                    {
                        // vii. Invoke “ValidateLocalIhiInformation” to determine if the patient has an IHI with unresolved alerts. If the response is neither OK nor InvalidIhi, then return.
                        ValidateLocalIhiInformation(patientMaster, response);
                        if (response.Status != HipsResponseIndicator.OK && response.Status != HipsResponseIndicator.InvalidIhi)
                        {
                            return response;
                        }

                        // viii. Invoke the “GetValidatedIhi” method of the “PatientIhiValidation” business logic. This will revalidate the IHI with the HI service if it was not validated within the configured period, 
                        // and it will set the IHI status to service unavailable if the IHI lookup fails. If the response is not OK then return.
                        PatientIhiValidation patientIhiValidation = new PatientIhiValidation();
                        IhiSearchResponse ihiSearchResponse = patientIhiValidation.GetValidatedIhi(registeredEnterprisePatient, hospital, User, patientMaster);

                        if (ihiSearchResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                        {
                            response.Status = ihiSearchResponse.HipsResponse.Status;
                            response.HipsErrorMessage = ihiSearchResponse.HipsResponse.HipsErrorMessage;
                            return response;
                        }

                        // ix. Invoke the “PcehrExists” method of the “DoesPcehrExist” business logic.
                        // This will obtain and store the PCEHR status for the new hospital’s HPI-O,
                        // and it will set the IHI status to service unavailable if the PCEHR lookup fails.
                        DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                        DoesPcehrExistResponse doesPcehrExistResponse = doesPcehrExist.PcehrExists(registeredEnterprisePatient, hospital, patientMaster, User);
                        if (doesPcehrExistResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                        {
                            response.Status = doesPcehrExistResponse.HipsResponse.Status;
                            response.HipsErrorMessage = doesPcehrExistResponse.HipsResponse.HipsErrorMessage;
                            return response;
                        }
                    }
                }
            }

            return response;
        }

        /// <summary>
        /// Finds the visitor hospital using either the internal hospital ID (if the supplied patient identifier is of PatientMasterId type) or the hospital code and hospital code system.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="hospital">The hospital.</param>
        /// <returns></returns>
        public HipsResponse GetVisitorHospital(PatientIdentifierBase patientIdentifier, out Hospital hospital)
        {
            if (HospitalSingleton.Value.IsHospitalListEmpty())
            {
                // Try to recover from database being down on HIPS startup.
                ListSingleton.Instance.Refresh(force: true);
            }

            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            PatientMasterId patientMasterId = patientIdentifier as PatientMasterId;
            if (patientMasterId != null && patientMasterId.HospitalId.HasValue)
            {
                hospital = HospitalSingleton.Value.FindVisitorHospital(patientMasterId.HospitalId.Value);
            }
            else
            {
                hospital = HospitalSingleton.Value.FindVisitorHospital(patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            }

            if (hospital == null)
            {
                if (HospitalSingleton.Value.IsHospitalListEmpty())
                {
                    response.Status = HipsResponseIndicator.DatabaseError;
                    response.HipsErrorMessage = ResponseStrings.UnableToGetHospitalList;
                }
                else
                {
                    response.Status = HipsResponseIndicator.InvalidHospital;
                    string notfound = patientMasterId != null && patientMasterId.HospitalId.HasValue 
                        ? string.Format(ResponseStrings.VisitorHospitalNotFound, patientIdentifier.HospitalCode, patientMasterId.HospitalId)
                        : string.Format(ResponseStrings.VisitorHospitalNotFoundByCode, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
                    response.HipsErrorMessage = notfound;
                }
                return response;
            }
            return response;
        }

        /// <summary>
        /// Get list of all non cancelled episodes for a patient
        /// </summary>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <param name="admissionDateStart">The admission date start for the query</param>
        /// <param name="admissionDateEnd">The admission date end for the query</param>
        /// <returns>
        /// A list of episodes for the patient with the optional admission start and end date, or null if none matched
        /// </returns>
        /// <exception cref="System.Exception">If a database error occurs while updating or inserting a stub episode</exception>
        public List<Episode> ListEpisodesWithoutCancelled(HospitalPatient hospitalPatient, DateTime? admissionDateStart, DateTime? admissionDateEnd)
        {
            List<Episode> episodes = EpisodeDataAccess.GetAll(hospitalPatient.PatientId, null, true);
            return episodes;
        }

        /// <summary>
        /// Reads, creates or updates a patient master and hospital patient for the given validated IHI information.
        /// </summary>
        /// <param name="validatedIhi">The IHI along with its last validated date and demographic information for revalidation</param>
        /// <param name="hospital">The hospital</param>
        /// <param name="hospitalPatient">output of hospital patient</param>
        /// <param name="patientMaster">output of patient master</param>
        /// <returns>Information about success or failure</returns>
        public HipsResponse ReadCreateOrUpdatePatient(ValidatedIhi validatedIhi, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            hospitalPatient = null;
            patientMaster = null;
            SqlTransaction transaction = null;
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                // Validate the ihi information
                Sex sex;
                response = ValidateValidatedIhiInformation(validatedIhi, out sex);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    return response;
                }

                //populate the patient master from the ihi provided
                PatientMasterDataAccess.GetByIhi(validatedIhi.Ihi, hospital.HealthProviderOrganisationNetworkId, out patientMaster);

                using (SqlCommand command = base.Command)
                {
                    transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    // Insert the patientmaster if it does not already exist
                    if (!patientMaster.PatientMasterId.HasValue)
                    {
                        patientMaster.DateOfBirth = validatedIhi.DateOfBirth;
                        patientMaster.CurrentSexId = sex.SexId.Value;
                        if (!PatientMasterDataAccess.Insert(patientMaster, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, PatientMasterDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    // Add in all IHI info to new or gathered patientmaster item
                    patientMaster.Ihi = validatedIhi.Ihi;
                    patientMaster.IhiStatusId = (int)validatedIhi.IhiStatus;
                    patientMaster.IhiRecordStatusId = (int)validatedIhi.IhiRecordStatus;
                    patientMaster.IhiLastValidated = validatedIhi.IhiLastValidated;
                    patientMaster.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;

                    //if the DOB has changed then we need to create a flag so that we know to process an update
                    bool patientMasterDOBUpdated = (patientMaster.DateOfBirth != validatedIhi.DateOfBirth);
                    if (!Settings.Instance.RegisteredDateOfBirthEnabled)
                    {
                        //now we can update the value after determining if it has been updated
                        patientMaster.DateOfBirth = validatedIhi.DateOfBirth;
                    }

                    patientMaster.CurrentSexId = sex.SexId.Value;
                    bool patientMasterNameUpdated = patientMaster.SetNewCurrentName(null, validatedIhi.GivenName, validatedIhi.FamilyName, null);

                    // Get the ihi information for the patient
                    PatientMasterIhi patientMasterIhi;
                    IhiDataAccess.Get(patientMaster.PatientMasterId.Value, hospital.HealthProviderOrganisationNetworkId, out patientMasterIhi);

                    //if any of the values for the PatientMasterIhi object have been changed then update it
                    if ((patientMasterIhi.Ihi != validatedIhi.Ihi) ||
                            (patientMasterIhi.DateLastValidated != validatedIhi.IhiLastValidated) ||
                            (patientMasterIhi.IhiStatusId != (int)validatedIhi.IhiStatus) ||
                            (patientMasterIhi.IhiRecordStatusId != (int)validatedIhi.IhiRecordStatus) ||
                            (patientMasterIhi.RegisteredFamilyName != validatedIhi.FamilyName) ||
                            (patientMasterIhi.RegisteredGivenName != validatedIhi.GivenName) ||
                            (patientMasterIhi.RegisteredSexId != sex.SexId.Value) ||
                            (patientMasterDOBUpdated))
                    {
                        // Populate with the passed in values ready for insert or update
                        patientMasterIhi.PatientMasterId = patientMaster.PatientMasterId.Value;
                        patientMasterIhi.Ihi = validatedIhi.Ihi;
                        patientMasterIhi.DateLastValidated = validatedIhi.IhiLastValidated;
                        patientMasterIhi.IhiStatusId = (int)validatedIhi.IhiStatus;
                        patientMasterIhi.IhiRecordStatusId = (int)validatedIhi.IhiRecordStatus;
                        patientMasterIhi.RegisteredFamilyName = validatedIhi.FamilyName;
                        patientMasterIhi.RegisteredGivenName = validatedIhi.GivenName;
                        patientMasterIhi.RegisteredSexId = sex.SexId.Value;
                        patientMasterIhi.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;
                        patientMasterIhi.DateOfBirth = validatedIhi.DateOfBirth;

                        if (Settings.Instance.RegisteredDateOfBirthEnabled)
                        {
                            patientMasterIhi.RegisteredDateOfBirth = validatedIhi.DateOfBirth;
                        }

                        // Insert or Update the patient master ihi
                        if (!IhiDataAccess.Update(patientMasterIhi, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, IhiDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    //if any of the PatientMaster object has changed then update it
                    if ((patientMaster.Ihi != validatedIhi.Ihi) ||
                            (patientMaster.IhiStatusId != (int)validatedIhi.IhiStatus) ||
                            (patientMaster.IhiRecordStatusId != (int)validatedIhi.IhiRecordStatus) ||
                            (patientMaster.IhiLastValidated != validatedIhi.IhiLastValidated) ||
                            (patientMasterDOBUpdated) ||
                            (patientMaster.CurrentSexId != sex.SexId.Value) ||
                            (patientMasterNameUpdated)
                        )
                    {
                        //update the patient master with added information or changes as passed through
                        if (!PatientMasterDataAccess.Update(patientMaster, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, PatientMasterDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    //get the active hospital patient and if it does not exists then insert
                    HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
                    if (hospitalPatient == null || !hospitalPatient.PatientId.HasValue)
                    {
                        // This jurisdiction doesn't want to provide hospital MRNs so we will generate a dummy one.
                        hospitalPatient = new HospitalPatient();
                        hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                        hospitalPatient.HospitalId = hospital.HospitalId.Value;
                        hospitalPatient.Mrn = System.Guid.NewGuid().ToString().Substring(0, 20);  //Dummy MRN
                        if (!HospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, HospitalPatientDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    transaction.Commit();
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                if (transaction != null)
                {
                    transaction.Rollback();
                    if (transaction.Connection != null)
                    {
                        transaction.Connection.Close();
                    }
                }
                EventLogger.WriteLog(ResponseStrings.UnableToInsertPatientStub, ex, User, LogMessage.HIPS_MESSAGE_066);
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = ResponseStrings.UnableToInsertPatientStub;
            }
            return response;
        }

        /// <summary>
        /// Validates the locally-stored IHI information for the patient before access to the PCEHR.
        /// This does not go to the Medicare HI Service. It checks whether an IHI is assigned to the
        /// patient, that it is 16 digits, that the record status is Verified, that the status is
        /// Active or Deceased, and that there are no unresolved alerts.
        /// </summary>
        /// <param name="patientMaster">The local patient demographics</param>
        /// <param name="response">Stores information about validation success or failure</param>
        public void ValidateLocalIhiInformation(PatientMaster patientMaster, HipsResponse response)
        {
            // Need to check status first because outstanding alerts should trump invalid IHI.
            // If the alert statuses are present, we return immediately.
            switch ((IhiStatus)patientMaster.IhiStatusId)
            {
                case IhiStatus.DuplicateIhi:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusDuplicateIhi;
                    return;

                case IhiStatus.DuplicatePatient:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusDuplicatePatient;
                    return;

                case IhiStatus.MergeConflict:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusMergeConflict;
                    return;

                case IhiStatus.MedicareDvaChangeMismatch:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusMedicareDvaChangeMismatch;
                    return;

                case IhiStatus.DemographicMismatch:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusDemographicMismatch;
                    return;

                case IhiStatus.Active:
                    response.Status = HipsResponseIndicator.OK; // OK
                    break;

                case IhiStatus.Deceased:
                    response.Status = HipsResponseIndicator.OK; // Can still access PCEHR
                    break;

                case IhiStatus.Expired:
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusExpired;
                    break;

                case IhiStatus.Resolved:

                    // When an IHI is resolved, the new active one is stored for
                    // the patient, and the old resolved one is stored in the
                    // history. There should never be an IHI with this status on
                    // a patient.
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusResolved;
                    break;

                case IhiStatus.Retired:

                    // When an IHI is Retired, it stays attached
                    // to the patient, marked with this status.
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusRetired;
                    return;

                case IhiStatus.ServiceUnavailable:
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusServiceUnavailable;
                    break;

                case IhiStatus.Unknown:
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusUnknown;
                    break;

                default:
                    response.Status = HipsResponseIndicator.SystemError;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusNotHandled;
                    break;
            }
            if (string.IsNullOrEmpty(patientMaster.Ihi))
            {
                response.Status = HipsResponseIndicator.InvalidIhi;
                response.HipsErrorMessage = ResponseStrings.IhiNotSupplied;
                return;
            }
            if (patientMaster.Ihi.Length != 16)
            {
                response.Status = HipsResponseIndicator.InvalidIhi;
                response.HipsErrorMessage = ResponseStrings.IhiLengthNot16;
                return;
            }
            if (patientMaster.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                response.Status = HipsResponseIndicator.InvalidIhi;
                response.HipsErrorMessage = ResponseStrings.IhiRecordStatusNotVerified;
                return;
            }
        }

        /// <summary>
        /// Checks that the family and given names were provided and the sex code was a recognised code.
        /// Also checks that the IHI is 16 digits and the record status is Verified.
        /// </summary>
        /// <param name="validatedIhi">The supplied validated IHI information</param>
        /// <param name="sex">Outputs the parsed sex code</param>
        /// <returns>Whether the information passed validation</returns>
        private HipsResponse ValidateValidatedIhiInformation(ValidatedIhi validatedIhi, out Sex sex)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            sex = null;
            if (string.IsNullOrEmpty(validatedIhi.FamilyName) || string.IsNullOrEmpty(validatedIhi.GivenName))
            {
                response.Status = HipsResponseIndicator.InvalidPatient;
                response.HipsErrorMessage = ResponseStrings.NameNotSupplied;
                return response;
            }
            sex = ListSingleton.Instance.AllSexes.Where(a => a.SexId == (int)validatedIhi.Sex).FirstOrDefault();
            if (sex == null)
            {
                response.Status = HipsResponseIndicator.InvalidPatient;
                response.HipsErrorMessage = ResponseStrings.InvalidSex;
                return response;
            }
            return response;
        }

        #endregion Methods

        #region Private

        /// <summary>
        /// Creates the patient from demographic data.
        /// </summary>
        /// <param name="demographic">The demographic.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="patientMaster">The patient master.</param>
        public void CreatePatient(Demographic demographic, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            using (SqlCommand command = base.Command)
            {
                try
                {
                    SqlTransaction transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    hospitalPatient = new HospitalPatient();
                    patientMaster = new PatientMaster();

                    // Store the key demographic information into the patient master record
                    patientMaster.SetNewCurrentName(-1, demographic.GivenName, demographic.FamilyName, -1);
                    patientMaster.MedicareNumber = demographic.MedicareNumber;
                    patientMaster.MedicareIrn = demographic.MedicareIrn;
                    patientMaster.IsMedicareNumberValid = Medicare.Validate(patientMaster.MedicareNumber);
                    patientMaster.DvaNumber = demographic.DvaNumber;
                    patientMaster.DateOfBirth = demographic.DateOfBirth;
                    patientMaster.RegisteredDateOfBirth = demographic.DateOfBirth;
                    patientMaster.CurrentSexId = (int)demographic.Sex;

                    if (!PatientMasterDataAccess.Insert(patientMaster, transaction))
                    {
                        // Failed
                    }

                    // Create a new hospital patient record with a generated MRN
                    hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                    hospitalPatient.HospitalId = hospital.Id.Value;
                    hospitalPatient.Mrn = ("AR" + System.Guid.NewGuid().ToString()).Substring(0, 20);

                    if (!HospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                    {
                        // Failed
                    }

                    transaction.Commit();
                }
                finally
                {
                    command.Connection.Close();
                }
            }
        }

        /// <summary>
        /// Look ups a patient by the demographics within a hospital.
        /// </summary>
        /// <param name="familyName">Name of the family.</param>
        /// <param name="givenNames">The given names.</param>
        /// <param name="dateOfBirth">The date of birth.</param>
        /// <param name="sexId">The sex identifier.</param>
        /// <param name="medicareCardNumber">The Medicare card number.</param>
        /// <param name="medicareIrn">The Medicare IRN.</param>
        /// <param name="dvaFileNumber">The DVA file number.</param>
        /// <param name="patientMaster">The patient master.</param>
        private void LookupByDemographicsWithinHospital(string familyName, string givenNames, DateTime? dateOfBirth, int? sexId, string medicareCardNumber, string medicareIrn, string dvaFileNumber, Hospital hospital, out PatientMaster patientMaster)
        {
            patientMaster = null;
            bool? medicareIsValid = Medicare.Validate(medicareCardNumber);
            if (!medicareIsValid.HasValue || !medicareIsValid.Value)
            {
                medicareCardNumber = null;
                medicareIrn = null;
            }
            if (dateOfBirth.HasValue && sexId.HasValue && (medicareCardNumber != null || dvaFileNumber != null))
            {
                PatientMasterDataAccess.GetByDemographicsWithinHospital(familyName, givenNames, dateOfBirth.Value, sexId.Value, medicareCardNumber, medicareIrn, dvaFileNumber, hospital, out patientMaster);
            }
        }

        #endregion Private
    }
}