﻿using System.Runtime.Serialization;
using System.Text;

namespace HIPS.CommonSchemas
{
    /// <summary>
    /// Standardised HIPS Response Message
    /// </summary>
    [KnownType(typeof(HipsResponse))]
    [DataContract]
    public class HipsResponse
    {
        #region Private Members

        /// <summary>
        /// HIPS error message.
        /// </summary>
        private string hipsErrorMessage;

        /// <summary>
        /// Response code.
        /// </summary>
        private string responseCode;

        /// <summary>
        /// Response code description.
        /// </summary>
        private string responseCodeDescription;

        /// <summary>
        /// Response code details.
        /// </summary>
        private string responseCodeDetails;

        /// <summary>
        /// Status indicator.
        /// </summary>
        private HipsResponseIndicator status;

        #endregion Private Members

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="HipsResponse"/> class.
        /// </summary>
        /// <param name="status">The status.</param>
        /// <param name="hipsErrorMessage">The error message.</param>
        public HipsResponse(HipsResponseIndicator status, string hipsErrorMessage = null)
        {
            this.status = status;
            this.hipsErrorMessage = hipsErrorMessage;
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="HipsResponse"/> class.
        /// </summary>
        public HipsResponse()
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="HipsResponse"/> class.
        /// </summary>
        /// <param name="ex">Any exception.</param>
        public HipsResponse(System.Exception ex)
        {
            this.Status = HipsResponseIndicator.SystemError;
            this.HipsErrorMessage = ex.Message;
            this.ResponseCode = ex.GetType().Name;
            this.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
            this.ResponseCodeDescription = ex.InnerException != null ? ex.InnerException.GetType().Name : null;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the HIPS error message.
        /// </summary>
        [DataMember]
        public string HipsErrorMessage
        {
            get
            {
                return this.hipsErrorMessage;
            }

            set
            {
                this.hipsErrorMessage = value;
            }
        }

        /// <summary>
        /// Gets or sets the response code.
        /// </summary>
        [DataMember]
        public string ResponseCode
        {
            get
            {
                return this.responseCode;
            }

            set
            {
                this.responseCode = value;
            }
        }

        /// <summary>
        /// Gets or sets the response code description.
        /// </summary>
        [DataMember]
        public string ResponseCodeDescription
        {
            get
            {
                return this.responseCodeDescription;
            }

            set
            {
                this.responseCodeDescription = value;
            }
        }

        /// <summary>
        /// Gets or sets the response code details.
        /// </summary>
        [DataMember]
        public string ResponseCodeDetails
        {
            get
            {
                return this.responseCodeDetails;
            }

            set
            {
                this.responseCodeDetails = value;
            }
        }

        /// <summary>
        /// Gets or sets the status indicator.
        /// </summary>
        [DataMember]
        public HipsResponseIndicator Status
        {
            get
            {
                return this.status;
            }

            set
            {
                this.status = value;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Creates a string representation of the HIPS response.
        /// </summary>
        /// <returns>A string representation of the HIPS response.</returns>
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder(this.status.ToString());
            if (!string.IsNullOrEmpty(this.responseCode))
            {
                sb.AppendFormat(", Code: {0}", this.responseCode);
            }
            if (!string.IsNullOrEmpty(this.responseCodeDescription))
            {
                sb.AppendFormat(", Description: {0}", this.responseCodeDescription);
            }
            if (!string.IsNullOrEmpty(this.responseCodeDetails))
            {
                sb.AppendFormat(", Details: {0}", this.responseCodeDetails);
            }
            if (!string.IsNullOrEmpty(this.hipsErrorMessage))
            {
                sb.AppendFormat(", Message: {0}", this.hipsErrorMessage);
            }
            return sb.ToString();
        }

        #endregion Methods
    }
}