﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Data.SqlTypes;
using System.Linq;
using HIPS.Base.Schemas.Enumerators;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.Enumerators;
using HIPS.HibIntegration.HL7.DataStructure;
using HIPS.HibIntegration.HL7.Segment;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using hips = HIPS.PcehrDataStore.Schemas.Schemas;

namespace HIPS.HibIntegration.Loader
{
    public class PatientLoader
    {
        #region Private Static Fields

        private const int MINIMUM_DATE_OF_BIRTH_LENGTH = 8;

        private const int MINIMUM_DATE_OF_BIRTH_YEAR = 1753;

        // Defaults to the Home code for addresstype
        private const string UnknownAddressType = "H";

        // Placeholder values for situations when information is unknown or not supplied.
        private const int UnknownForeignKey = -1;

        private readonly static Dictionary<string, SexEnumerator> SexCodeMapping = new Dictionary<string, SexEnumerator>
        {
            { "M", SexEnumerator.Male },
            { "F", SexEnumerator.Female },
            { "A", SexEnumerator.IntersexOrIndeterminate },
            { "O", SexEnumerator.IntersexOrIndeterminate },
            { "U", SexEnumerator.NotStatedOrInadequatelyDescribed },
        };

        #endregion Private Static Fields

        #region Private Instance Fields

        private HospitalPatientDl hospitalPatientDataAccess;
        private PatientMasterDl patientMasterDataAccess;
        private SqlTransaction transaction;
        private UserDetails user;

        #endregion Private Instance Fields

        public PatientLoader(UserDetails user, SqlTransaction transaction)
        {
            this.transaction = transaction;
            this.user = user;
            hospitalPatientDataAccess = new HospitalPatientDl(user);
            patientMasterDataAccess = new HIPS.PcehrDataStore.DataAccess.PatientMasterDl(user);
        }

        /// <summary>
        /// Gets the first identifier from the list that matches the given identifier.
        /// If there is no matching identifier, and a fallback item is given, then use the given fallback item.
        /// If an identifier is found and the type is a Medical Record Number ("MR") then apply the standard
        /// padding.
        /// </summary>
        /// <param name="identifierList">An array of identifiers such as PID-3 Patient Identifier List</param>
        /// <param name="type">The type of identifier that is required</param>
        /// <param name="fallback">The identifier to use if not found in the list</param>
        /// <returns>The matching identifier, or null if no matches</returns>
        public static CX IdentifierWithType(CX[] identifierList, IdentifierTypeCode type, CX fallback = null)
        {
            CX id = identifierList.Where(a => type.ToString() == a.identifiertypecode).FirstOrDefault();
            if (id == null)
            {
                id = fallback;
            }
            if (id != null && type == IdentifierTypeCode.MR)
            {
                id.ID = MrnPadding.Pad(id.ID);
            }
            return id;
        }

        /// <summary>
        /// Populates the patient demographic information from the HL7 message into the patient record.
        /// </summary>
        /// <param name="pid">The PID segment of the HL7 message.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="oldPatientMaster">Returns a clone of the existing patient master, for checking what has changed.</param>
        public void Populate(PID pid, Hospital hospital, HospitalPatient hospitalPatient, PatientMaster patientMaster, out PatientMaster oldPatientMaster)
        {
            if (patientMaster == null)
            {
                patientMaster = new PatientMaster();
            }
            oldPatientMaster = CloneKeyIhiCriteria(patientMaster);

            // PID-3 Patient Identifier List: Medicare Card Number and DVA File Number
            CX medicareCard = IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.MC);
            string medicareCardNumber = medicareCard == null ? null : medicareCard.ID;
            string medicareIrn = null;
            if (medicareCardNumber != null && medicareCardNumber.Length == 11)
            {
                medicareIrn = medicareCardNumber.Substring(10, 1);
                medicareCardNumber = medicareCardNumber.Substring(0, 10);
            }
            CX dvaFile = IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.DVA);
            string dvaFileNumber = dvaFile == null ? null : dvaFile.ID;

            // PID-7 Patient Date of Birth
            DateTime? dateOfBirth = null;
            try
            {
                dateOfBirth = pid.DateTimeOfBirth.TimestampValue;
            }
            catch (Exception)
            {
                string errorMessage = string.Format(ConstantsResource.InvalidDateOfBirth, pid.DateTimeOfBirth.timeofanevent);
                throw new HL7MessageInfoException(errorMessage);
            }
            if (string.IsNullOrEmpty(pid.DateTimeOfBirth.timeofanevent))
            {
                throw new HL7MessageInfoException(ConstantsResource.NoDateOfBirthForPatient);
            }
            if (pid.DateTimeOfBirth.timeofanevent.Length < MINIMUM_DATE_OF_BIRTH_LENGTH)
            {
                throw new HL7MessageInfoException(ConstantsResource.InsufficientPrecisionInDateOfBirth);
            }
            if (dateOfBirth.HasValue && dateOfBirth.Value.Year < MINIMUM_DATE_OF_BIRTH_YEAR)
            {
                string errorMessage = string.Format(ConstantsResource.DateOfBirthOutOfRange, dateOfBirth);
                throw new HL7MessageInfoException(errorMessage);
            }

            // PID-8 Sex: Enterprise Standard Table Gender: M, F, (A, O), U transformed to M, F, I, U
            SexEnumerator sexEnum = SexEnumerator.NotStatedOrInadequatelyDescribed;
            int? sexId = null;
            if (SexCodeMapping.TryGetValue(pid.Sex.identifier, out sexEnum))
            {
                sexId = (int)sexEnum;
            }

            // Store the key demographic information into the patient master record
            patientMaster.MedicareNumber = medicareCardNumber;
            patientMaster.MedicareIrn = medicareIrn;

            if (oldPatientMaster == null || oldPatientMaster.MedicareNumber != patientMaster.MedicareNumber || !patientMaster.IsMedicareNumberValid.HasValue)
            {
                patientMaster.IsMedicareNumberValid = Medicare.Validate(patientMaster.MedicareNumber);
            }

            patientMaster.DvaNumber = dvaFileNumber;

            if (dateOfBirth.HasValue)
            {
                patientMaster.DateOfBirth = dateOfBirth.Value;
            }
            patientMaster.CurrentSexId = sexId.HasValue ? sexId.Value : UnknownForeignKey;

            // Store the other information into the patient master record
            PopulateAddress(pid, patientMaster);
            PopulateSecondaryDemographics(pid, patientMaster, user);
            PopulateCommunicationMethods(pid, patientMaster);
            PopulateStatePatientIdentifier(pid, patientMaster, hospitalPatient, hospital, user);

            SavePatient(patientMaster, hospitalPatient);
        }

        /// <summary>
        /// Verifies that an existing patient has the same last name and date of birth as has been specified in
        /// the incoming message. Throws an HL7LoaderException if either are different.
        /// </summary>
        /// <param name="pid">PID segment of HL7 message</param>
        /// <param name="patientMaster">Existing patient master record</param>
        public void Verify(PID pid, PatientMaster patientMaster)
        {
            string message;
            string receivedLastName = pid.PatientName[0].familylastname.familyname;

            if (patientMaster.CurrentName.FamilyName.ToUpper() == receivedLastName.ToUpper())
            {
                // The last names match OK, check the dates of birth, ignoring any time component
                DateTime receivedDate = pid.DateTimeOfBirth.TimestampValue.Value.Date;
                DateTime currentDate = patientMaster.DateOfBirth.Date;

                if (receivedDate != currentDate)
                {
                    message = string.Format(ConstantsResource.DateOfBirthMismatchError, receivedDate, currentDate);
                    throw new HL7MessageErrorException(message);
                }
            }
            else
            {
                message = string.Format(ConstantsResource.LastNameMismatchError, receivedLastName, patientMaster.CurrentName.FamilyName);
                throw new HL7MessageErrorException(message);
            }
        }

        /// <summary>
        /// Populates the name.
        /// </summary>
        /// <param name="pid">The pid.</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateAddress(PID pid, PatientMaster patientMaster)
        {
            ListSingleton lists = ListSingleton.Instance;
            List<hips.Address> updatedAddresses = new List<hips.Address>();

            // PID-11 Patient Address
            IEnumerable<XAD> xads = new XAD[0];
            if (pid.PatientAddress != null)
            {
                xads = xads.Concat(pid.PatientAddress);
            }

            //patientMaster.Addresses = new List<PcehrDataStore.Schemas.Schemas.Address>();
            foreach (XAD xad in xads)
            {
                string addressLine1 = xad.streetaddress;
                string addressLine2 = xad.otherdesignation;
                string addressLocality = xad.city;
                string addressStateCd = xad.stateorprovince;
                string addressPostcode = xad.ziporpostalcode;
                string addressTypeCode = xad.addresstype ?? UnknownAddressType;
                string addressCountry = xad.country ?? string.Empty;

                AddressType addressType = lists.AllAddressTypes.FirstOrDefault(a => a.Code == addressTypeCode);
                if (addressType == null)
                {
                    addressType = lists.AllAddressTypes.FirstOrDefault(c => c.Code == ConstantsResource.DefaultAddressTypeCode);
                }
                State addressState = lists.AllStates.FirstOrDefault(a => a.Code == addressStateCd);

                // Look for country by code.
                Country country = lists.AllCountries.FirstOrDefault(c => c.Code == addressCountry);
                if (country == null)
                {
                    // Look for country by name, case insensitive.
                    country = lists.AllCountries.FirstOrDefault(c => c.Description.ToUpper() == addressCountry.ToUpper());
                }
                if (country == null)
                {
                    // Default to Australia (code 1101).
                    country = lists.AllCountries.FirstOrDefault(c => c.Code == ConstantsResource.DefaultCountryCode);
                }

                hips.Address patientAddress = new hips.Address();
                patientAddress.AddressLine1 = addressLine1;
                patientAddress.AddressLine2 = addressLine2;
                patientAddress.PlaceName = addressLocality;
                patientAddress.PostCode = addressPostcode;
                patientAddress.AustralianStateId = (addressState != null) ? addressState.Id : UnknownForeignKey;
                patientAddress.CountryId = country.CountryId ?? UnknownForeignKey;
                patientAddress.AddressTypeId = addressType == null ? UnknownForeignKey : addressType.Id.Value;
                patientAddress.AddressTypeDescription = addressType == null ? null : addressType.Description;

                hips.Address existingAddress = patientMaster.Addresses.Find(result => result.CompareKey == patientAddress.CompareKey);
                if (existingAddress == null)
                {
                    updatedAddresses.Add(patientAddress);
                }
                else
                {
                    //Force address to be modified, so we can delete the missing addresses
                    updatedAddresses.Add(existingAddress);
                }
            }
            patientMaster.Addresses = updatedAddresses;
        }

        /// <summary>
        /// Populates the communication methods.
        /// </summary>
        /// <param name="pid">The pid.</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateCommunicationMethods(PID pid, PatientMaster patientMaster)
        {
            // PID-13 Home Phone Number and PID-14 Business Phone Number
            IEnumerable<XTN> xtns = new XTN[0];
            if (pid.PhoneNumberHome != null) xtns = xtns.Concat(pid.PhoneNumberHome);
            if (pid.PhoneNumberBusiness != null) xtns = xtns.Concat(pid.PhoneNumberBusiness);
            List<Contact> updatedContacts = new List<Contact>();
            bool haveHomePhone = false;
            foreach (XTN xtn in xtns)
            {
                // Use all of the available fields just in case source system populates them
                string phoneNumber = string.Format(ConstantsResource.ContactDetailsFormat, xtn.phonenumber1, xtn.CountryCode, xtn.Areacitycode, xtn.Phonenumber, xtn.Extension, xtn.anytext).Trim();
                Contact contact = null;
                if (TelecommunicationEquipmentType.CellularPhone == xtn.telecommunicationequipmentty)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.PersonalMobile, null);
                }
                else if (TelecommunicationUseCode.NetworkAddress == xtn.telecommunicationusecode)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.PersonalEmail, null);
                }
                else if (TelecommunicationUseCode.PrimaryResidenceNumber == xtn.telecommunicationusecode)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.HomePhone, null);
                    haveHomePhone = true;
                }
                else if (TelecommunicationUseCode.WorkNumber == xtn.telecommunicationusecode)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.WorkPhone, null);
                }
                else if (!haveHomePhone)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.HomePhone, null);
                }
                if (contact == null)
                {
                    continue;
                }
                Contact existingContact = patientMaster.Contacts.Find(result => result.CompareKey == contact.CompareKey);
                if (existingContact == null)
                {
                    updatedContacts.Add(contact);
                }
                else
                {
                    updatedContacts.Add(existingContact);
                }
            }
            patientMaster.Contacts = updatedContacts;
        }

        /// <summary>
        /// Populates the secondary demographics.
        /// </summary>
        /// <param name="pid">The pid.</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateSecondaryDemographics(PID pid, PatientMaster patientMaster, UserDetails user)
        {
            ListSingleton lists = ListSingleton.Instance;
            bool current = true;
            foreach (XPN patientName in pid.PatientName)
            {
                PatientMasterName updatedName = new PatientMasterName();
                updatedName.FamilyName = patientName.familylastname.familyname;
                updatedName.GivenNames = string.Format("{0} {1}", patientName.givenname.Trim(), patientName.middleinitialorname.Trim()).Trim();

                string titleCode = patientName.prefix;
                Title title = null;
                if (!string.IsNullOrWhiteSpace(titleCode))
                {
                    title = lists.AllTitles.Where(a => a.Code.Equals(titleCode)).FirstOrDefault();
                    if (title == null)
                    {
                        TitleDl titleDl = new TitleDl();
                        title = new Title();
                        title.Code = titleCode;
                        title.Description = titleCode;
                        titleDl.Insert(title);
                    }
                }
                string suffixCode = patientName.suffix;
                Suffix suffix = null;
                if (!string.IsNullOrWhiteSpace(suffixCode))
                {
                    suffix = lists.AllSuffixes.Where(a => a.Code.Equals(suffixCode)).FirstOrDefault();
                    if (suffix == null)
                    {
                        SuffixDl suffixDl = new SuffixDl();
                        suffix = new Suffix();
                        suffix.Code = suffixCode;
                        suffix.Description = suffixCode;
                        suffixDl.Insert(suffix);
                    }
                }
                updatedName.TitleId = (title == null ? UnknownForeignKey : title.Id.Value);
                updatedName.SuffixId = (suffix == null ? UnknownForeignKey : suffix.Id.Value);

                if (current)
                {
                    patientMaster.SetNewCurrentName(updatedName.TitleId, updatedName.GivenNames, updatedName.FamilyName, updatedName.SuffixId);
                }
            }

            // PID-29 Patient Date of Death
            DateTime? dateOfDeath = pid.PatientDeathDateandTime == null ? null : pid.PatientDeathDateandTime.TimestampValue;

            //the SQL Server date is a datetime and thus can't accept dates before 1/1/1753.
            if (dateOfDeath != null)
            {
                //the SQL Server date is a datetime and thus can't accept dates before SqlDateTime.MinValue (1/1/1753). or after SqlDateTime.MinValue (31/12/9999)
                if (dateOfDeath < SqlDateTime.MinValue.Value || dateOfDeath > SqlDateTime.MaxValue.Value)
                {
                    //if date is too small then set date to minimum SQL Date and flag indicator as invalid date
                    dateOfDeath = SqlDateTime.MinValue.Value;
                    patientMaster.DeathIndicatorId = (int)DeathIndicator.InvalidDate;

                    //extract patient MRN and Facility code for INFO message
                    CX cx = PatientLoader.IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.MR);
                    string patientMRN;
                    string facilityCode;
                    if (cx != null)
                    {
                        patientMRN = cx.ID;
                        facilityCode = cx.assigningauthority.namespaceID;
                    }
                    else
                    {
                        patientMRN = "'MRN not found'";
                        facilityCode = "'Facility Code not found'";
                    }

                    //log as info message only
                    string description = string.Format(ConstantsResource.InfoMessageDateOfDeathBeforeSQLDateMin, patientMRN, facilityCode);
                    System.Exception exception = new Exception(string.Format(ConstantsResource.InfoMessageDateOfDeathBeforeSQLDateMinEx, pid.PatientDeathDateandTime.TimestampValue, SqlDateTime.MinValue.Value, DeathIndicator.InvalidDate));
                    EventLogger.WriteLog(description, exception, user, LogMessage.HIPS_MESSAGE_112);
                }
                else
                {
                    patientMaster.DeathIndicatorId = (int)DeathIndicator.ValidDate;
                }
            }
            else
            {
                //if date of death is null then so is the indicator
                patientMaster.DeathIndicatorId = null;
            }

            //set date of death to calculated or received date of death
            patientMaster.DateOfDeath = dateOfDeath;
        }

        /// <summary>
        /// Populates the state patient identifier, if it was empty.
        /// Throws an error if the state patient identifier changes.
        /// Note, changes in state patient identifier should have been
        /// handled by the PatientLinker class before reaching this point.
        /// </summary>
        /// <param name="pid">The PID segment.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="user">The user.</param>
        private static void PopulateStatePatientIdentifier(PID pid, PatientMaster patientMaster, HospitalPatient hospitalPatient, Hospital hospital, UserDetails user)
        {
            if (pid.PatientID != null && pid.PatientID.identifiertypecode == IdentifierTypeCode.SAUHI.ToString())
            {
                string statePatientId = pid.PatientID.ID;
                if (string.IsNullOrEmpty(patientMaster.StatePatientId))
                {
                    patientMaster.StatePatientId = statePatientId;
                }
                else if (patientMaster.StatePatientId != statePatientId)
                {
                    string message = string.Format(ConstantsResource.StatePatientIdentifierChange, hospitalPatient.Mrn, hospital.Description, patientMaster.StatePatientId, statePatientId);
                    throw new HL7MessageErrorException(message);
                }
            }
        }

        /// <summary>
        /// Clones the key IHI criteria.
        /// </summary>
        /// <param name="old">The existing patient master.</param>
        /// <returns>A copy of the patient master which is independent of the given one that will be updated.</returns>
        private PatientMaster CloneKeyIhiCriteria(PatientMaster old)
        {
            if (!old.PatientMasterId.HasValue)
            {
                return null;
            }
            PatientMaster clone = new PatientMaster();
            clone.Names = (from name in old.Names
                           select new PatientMasterName()
                           {
                               FamilyName = name.FamilyName,
                               GivenNames = name.GivenNames,
                               TitleId = name.TitleId,
                               SuffixId = name.SuffixId,
                               NameTypeId = name.NameTypeId
                           }).ToList();
            clone.CurrentSexId = old.CurrentSexId;
            clone.RegisteredSexId = old.RegisteredSexId;
            clone.MedicareNumber = old.MedicareNumber;
            clone.MedicareIrn = old.MedicareIrn;
            clone.DvaNumber = old.DvaNumber;
            clone.DateOfBirth = old.DateOfBirth;
            clone.RegisteredFamilyName = old.RegisteredFamilyName;
            clone.RegisteredGivenName = old.RegisteredGivenName;
            return clone;
        }

        /// <summary>
        /// Attempts to look up an existing patient master using the
        /// demographic information of this new patient record.
        ///
        /// This is required because not all sites are integrated with EMPI.
        /// This will help to reduce the number of duplicate IHI alerts that
        /// need to be managed by the support team by manually triggering a
        /// merge.
        ///
        /// This looks for an existing patient master with the same first and
        /// last name, birth date, sex and either a Medicare Card Number or
        /// DVA File Number.
        /// </summary>
        /// <param name="familyName">Family name of the patient</param>
        /// <param name="givenNames">Given names of the patient</param>
        /// <param name="dateOfBirth">Date of Birth of the Patient</param>
        /// <param name="sexId">Sex of the patient</param>
        /// <param name="medicareCardNumber">Medicare card number of the patient</param>
        /// <param name="medicareIrn">Medicare individual reference number of the patient</param>
        /// <param name="dvaFileNumber">DVA file number of the patient</param>
        /// <param name="patientMaster">The patient master if found, else null</param>
        private void LookupByDemographics(string familyName, string givenNames, DateTime? dateOfBirth, int? sexId, string medicareCardNumber, string medicareIrn, string dvaFileNumber, out PatientMaster patientMaster)
        {
            patientMaster = null;
            bool? medicareIsValid = Medicare.Validate(medicareCardNumber);
            if (!medicareIsValid.HasValue || !medicareIsValid.Value)
            {
                medicareCardNumber = null;
                medicareIrn = null;
            }
            if (dateOfBirth.HasValue && sexId.HasValue && (medicareCardNumber != null || dvaFileNumber != null))
            {
                patientMasterDataAccess.GetByDemographics(familyName, givenNames, dateOfBirth.Value, sexId.Value, medicareCardNumber, medicareIrn, dvaFileNumber, out patientMaster);
            }
        }

        /// <summary>
        /// Saves the patient master and then the hospital patient, ensuring the hospital patient points to the patient master.
        /// </summary>
        /// <param name="patientMaster">Patient master object</param>
        /// <param name="hospitalPatient">Hospital patient object</param>
        /// <exception cref="HL7MessageErrorException">Unable to load the patient from the PID segment</exception>
        private void SavePatient(PatientMaster patientMaster, HospitalPatient hospitalPatient)
        {
            // Save the PatientMaster object.
            if (!patientMaster.PatientMasterId.HasValue)
            {
                if (!patientMasterDataAccess.Insert(patientMaster, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientMasterDataAccess.GetType().FullName));
                }
            }
            else
            {
                if (!patientMasterDataAccess.Update(patientMaster, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientMasterDataAccess.GetType().FullName));
                }
            }

            // We need to get the PatientMasterId and store it in the HospitalPatient before saving
            // the HospitalPatient.
            hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;

            // Save the HospitalPatient object.
            if (!hospitalPatient.PatientId.HasValue)
            {
                if (!hospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, hospitalPatientDataAccess.GetType().FullName));
                }
            }
            else
            {
                if (!hospitalPatientDataAccess.Update(hospitalPatient, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, hospitalPatientDataAccess.GetType().FullName));
                }
            }
        }
    }
}