﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the ClinicalDocumentVersion table
    /// </summary>
    public class ClinicalDocumentVersionDl : BaseDl
    {
        #region Constructors

        public ClinicalDocumentVersionDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets all.
        /// </summary>
        /// <returns></returns>
        public List<ClinicalDocumentVersion> GetAllVersions(int clinicalDocumentId)
        {
            List<ClinicalDocumentVersion> results = new List<ClinicalDocumentVersion>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentVersionGet"))
                {
                    AddIntParameter("ClinicalDocumentId", clinicalDocumentId, command);
                    AddBoolParameter("LatestVersionOnly", false, command);
                    results = GetPopulatedBusinessList<ClinicalDocumentVersion>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string description = string.Format(ConstantsResource.ErrorMessageClinicalDocumentVersionGet, clinicalDocumentId);
                EventLogger.WriteLog(description, ex, User, LogMessage.HIPS_MESSAGE_015);
            }
            return results;
        }

        /// <summary>
        /// Gets the latest version of a document.
        /// </summary>
        /// <param name="clinicalDocumentId">The document id.</param>
        /// <param name="item">Outputs the clinical document version</param>
        /// <returns>Whether the version was retrieved</returns>
        public bool GetLatestVersion(int clinicalDocumentId, out ClinicalDocumentVersion item)
        {
            bool result = false;
            item = new ClinicalDocumentVersion();

            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentVersionGet"))
                {
                    AddIntParameter("ClinicalDocumentId", clinicalDocumentId, command);
                    AddBoolParameter("LatestVersionOnly", true, command);
                    PopulateBusinessObject<ClinicalDocumentVersion>(command.ExecuteReader(), item);
                    command.Connection.Close();
                    result = true;
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentVersionGet, clinicalDocumentId), ex, User, LogMessage.HIPS_MESSAGE_016);
            }
            return result;
        }

        /// <summary>
        /// Gets the document version with a given source system document ID.
        /// </summary>
        /// <param name="sourceSystemDocumentId">The source system document id.</param>
        /// <param name="item">Outputs the clinical document version</param>
        /// <returns>Whether the version was retrieved</returns>
        public bool GetVersion(string sourceSystemDocumentId, out ClinicalDocumentVersion item)
        {
            bool result = false;
            item = new ClinicalDocumentVersion();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentVersionGet"))
                {
                    AddStringParameter("SourceSystemDocumentId", sourceSystemDocumentId, command);
                    AddBoolParameter("LatestVersionOnly", true, command);
                    PopulateBusinessObject<ClinicalDocumentVersion>(command.ExecuteReader(), item);
                    command.Connection.Close();
                    result = true;
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentVersionGet, sourceSystemDocumentId), ex, User, LogMessage.HIPS_MESSAGE_017);
            }
            return result;
        }

        /// <summary>
        /// Saves the specified ClinicalDocumentVersion.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocumentVersion.</param>
        /// <returns>Whether the item was saved</returns>
        public bool Save(ClinicalDocumentVersion clinicalDocumentVersion, SqlTransaction transaction = null)
        {
            if (clinicalDocumentVersion.ClinicalDocumentVersionId.HasValue)
            {
                return Update(clinicalDocumentVersion, transaction);
            }
            else
            {
                return Insert(clinicalDocumentVersion, transaction);
            }
        }

        /// <summary>
        /// Inserts the specified ClinicalDocumentVersion.
        /// </summary>
        /// <param name="clinicalDocumentVersion">The ClinicalDocumentVersion.</param>
        /// <returns></returns>
        private bool Insert(ClinicalDocumentVersion clinicalDocumentVersion, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentVersionInsert", transaction))
            {
                return base.Insert<ClinicalDocumentVersion>(clinicalDocumentVersion, command);
            }
        }

        /// <summary>
        /// Updates the specified ClinicalDocumentVersion.
        /// </summary>
        /// <param name="clinicalDocumentVersion">The ClinicalDocumentVersion.</param>
        /// <returns></returns>
        private bool Update(ClinicalDocumentVersion clinicalDocumentVersion, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentVersionUpdate", transaction))
            {
                return base.Update<ClinicalDocumentVersion>(clinicalDocumentVersion, command);
            }
        }

        #endregion Methods
    }
}