﻿using System;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using BaseEnumerators = HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents the name of a patient
    /// </summary>
    [KnownType(typeof(PatientMasterName))]
    [Serializable]
    [DataContract]
    public class PatientMasterName : BaseSchema
    {
        #region Private Members

        private int? titleId;
        private string title;
        private string givenNames;
        private string familyName;
        private int? suffixId;
        private string suffix;
        private int nameTypeId;
        private string nameType;

        #endregion Private Members

        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="PatientMasterName"/> class.
        /// </summary>
        public PatientMasterName()
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PatientMasterName"/> class.
        /// </summary>
        /// <param name="titleId">The title id.</param>
        /// <param name="title">The title.</param>
        /// <param name="givenNames">The given names.</param>
        /// <param name="familyName">Name of the family.</param>
        /// <param name="suffixId">The suffix id.</param>
        /// <param name="suffix">The suffix.</param>
        /// <param name="nameTypeId">The name type id.</param>
        /// <param name="nameType">Type of the name.</param>
        public PatientMasterName(int? titleId, string title, string givenNames, string familyName, int? suffixId, string suffix, int nameTypeId, string nameType)
        {
            this.titleId = titleId;
            this.title = title;
            this.givenNames = givenNames;
            this.familyName = familyName;
            this.suffixId = suffixId;
            this.suffix = suffix;
            this.nameTypeId = nameTypeId;
            this.nameType = nameType;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the patient master name id.
        /// </summary>
        /// <value>
        /// The patient master name id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.AutoGeneratedKey)]
        public int? PatientMasterNameId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        /// <summary>
        /// Gets or sets the patient master id.
        /// </summary>
        /// <value>
        /// The patient master id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int PatientMasterId { get; set; }

        /// <summary>
        /// Gets or sets the title id.
        /// </summary>
        /// <value>
        /// The title id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int? TitleId
        {
            get
            {
                return titleId;
            }
            set
            {
                titleId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => TitleId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the title.
        /// </summary>
        /// <value>
        /// The title.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string Title
        {
            get
            {
                return title;
            }
            set
            {
                title = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => Title);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the given names.
        /// </summary>
        /// <value>
        /// The given names.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string GivenNames
        {
            get
            {
                return givenNames;
            }
            set
            {
                givenNames = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => GivenNames);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the name of the family.
        /// </summary>
        /// <value>
        /// The name of the family.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string FamilyName
        {
            get
            {
                return familyName;
            }
            set
            {
                familyName = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => FamilyName);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the suffix id.
        /// </summary>
        /// <value>
        /// The suffix id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int? SuffixId
        {
            get
            {
                return suffixId;
            }
            set
            {
                suffixId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => SuffixId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the suffix.
        /// </summary>
        /// <value>
        /// The suffix.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string Suffix
        {
            get
            {
                return suffix;
            }
            set
            {
                suffix = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => Suffix);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the name type id.
        /// </summary>
        /// <value>
        /// The name type id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int NameTypeId
        {
            get
            {
                return nameTypeId;
            }
            set
            {
                nameTypeId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => NameTypeId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the type of the name.
        /// </summary>
        /// <value>
        /// The type of the name.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string NameType
        {
            get
            {
                return nameType;
            }
            set
            {
                nameType = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => NameType);
#endif
            }
        }

        /// <summary>
        /// Gets the compare key.
        /// </summary>
        public string CompareKey
        {
            get
            {
                string familyName = (FamilyName ?? string.Empty).Trim().ToUpper();
                string givenNames = (GivenNames ?? string.Empty).Trim().ToUpper();

                return string.Format("{0}{1}{2}{3}", familyName, givenNames, SuffixId, TitleId);
            }
        }

        /// <summary>
        /// Returns a <see cref="System.String"/> that represents this instance.
        /// </summary>
        /// <returns>
        /// A <see cref="System.String"/> that represents this instance.
        /// </returns>
        public override string ToString()
        {
            return string.Format("{0}, {1} {2} {3}", this.familyName, this.title, this.givenNames, this.suffix).Trim();
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Creates a copy of this PatientMasterName to assign to a new patient master.
        /// </summary>
        /// <returns>A copy of this PatientMasterName.</returns>
        internal PatientMasterName CloneForNewPatientMaster()
        {
            PatientMasterName clone = this.MemberwiseClone() as PatientMasterName;
            clone.PatientMasterId = 0;
            clone.PatientMasterNameId = null;
            return clone;
        }

        #endregion Methods
    }
}