﻿using System;
using System.Runtime.Serialization;

using EntLibVal = Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.ServiceContracts.Common.DTO.PatientIdentifier
{
    /// <summary>
    /// National IHI number that has been validated against the HI Service by another system.
    /// </summary>
    [KnownType(typeof(PatientIdentifierBase))]
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public class ValidatedIhi : PatientIdentifierBase
    {
        #region Properties

        /// <summary>
        /// Gets or sets the date of birth of the patient. This date of birth
        /// must have been used in a successful IHI search in the HI Service.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DateTimeRangeValidator(Constants.SupportedDates.MINIMUM, Constants.SupportedDates.MAXIMUM)]
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the family name of the patient. This family name must
        /// have been used in a successful IHI search in the HI Service.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.StringLengthValidator(1, 80)]
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets the given names of the patient. There may be multiple
        /// given names separated by spaces. One of them must be the given name
        /// that was used in a successful IHI search in the HI Service.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.StringLengthValidator(1, 80)]
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets the IHI.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.StringLengthValidator(16, 16)]
        public string Ihi { get; set; }

        /// <summary>
        /// Gets or sets the date and time when the IHI was last validated using
        /// a search of the HI Service.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DateTimeRangeValidator(Constants.SupportedDates.MINIMUM, Constants.SupportedDates.MAXIMUM)]
        public DateTime IhiLastValidated { get; set; }

        /// <summary>
        /// Gets or sets the IHI record status.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DomainValidator(IhiRecordStatus.None, Negated = true)]
        public IhiRecordStatus IhiRecordStatus { get; set; }

        /// <summary>
        /// Gets or sets the IHI status.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DomainValidator(IhiStatus.None, Negated = true)]
        public IhiStatus IhiStatus { get; set; }

        /// <summary>
        /// Gets or sets the sex of the patient. This sex must have been used
        /// in a successful IHI search in the HI Service.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DomainValidator(Sex.None, Negated = true)]
        public Sex Sex { get; set; }

        #endregion Properties
    }
}