﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using System.Data.SqlServerCe;
using System.IO;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Common.DTO;
using HIPS.ServiceContracts.Common.DTO.ParticipatingIndividual;
using HIPS.ServiceContracts.Common.DTO.UserIdentity;
using HIPS.ServiceContracts.Pcehr.Message;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.Helpers;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// Provides patients for use in CCA testing.
    /// </summary>
    public class CcaPatient : PatientTestHelpersBase
    {

        #region Configuration Properties

        /// <summary>
        /// Sets the override hospital ID.
        /// </summary>
        public static int? OverrideHospitalId { private get; set; }

        /// <summary>
        /// The patient identifier type.
        /// </summary>
        public static PatientIdentifierType PatientIdentifierType
        {
            get
            {
                return (PatientIdentifierType)Enum.Parse(typeof(PatientIdentifierType), ConfigurationManager.AppSettings["PatientIdentifierType"].ToString());
            }
        }

        /// <summary>
        /// Which of the rows in the CcaTestPatients table TestPcehr has the data for this patient.
        /// </summary>
        public string TestDataId
        {
            get;
            set;
        }

        /// <summary>
        /// Whether to set the Medicare number to an invalid value.
        /// </summary>
        public bool HasInvalidMedicareNumber { get; set; }

        /// <summary>
        /// Whether to set the Medicare number to null.
        /// </summary>
        public bool HasNoMedicareNumber { get; set; }

        private SqlCeConnection ccaTestPatientsConnectionField;

        /// <summary>
        /// A connection to the SQL Compact database containing the test patients.
        /// </summary>
        private SqlCeConnection CcaTestPatientsConnection
        {
            get
            {
                if (ccaTestPatientsConnectionField == null)
                {
                    string connectionString = ConfigurationManager.ConnectionStrings["CcaTestPatientsConnectionString"].ConnectionString;
                    ccaTestPatientsConnectionField = new SqlCeConnection(connectionString);
                    ccaTestPatientsConnectionField.Open();
                }
                return ccaTestPatientsConnectionField;
            }
        }

        private SqlCeCommand GetCcaTestPatientsCommand()
        {
            return CcaTestPatientsConnection.CreateCommand();
        }

        private Dictionary<string, string> testPcehrField;

        private Dictionary<string, string> TestPcehr
        {
            get
            {
                if (testPcehrField == null)
                {
                    testPcehrField = new Dictionary<string, string>();
                    using (SqlCeCommand c = GetCcaTestPatientsCommand())
                    {
                        c.CommandText = string.Format("SELECT * FROM TestPcehr WHERE TestDataId='{0}'", TestDataId);
                        SqlCeResultSet resultSet = c.ExecuteResultSet(ResultSetOptions.None);
                        bool gotRow = resultSet.Read();
                        Assert.IsTrue(gotRow, DialogueResource.TestDataIdNotFound, TestDataId);
                        for (int i = 0; i < resultSet.FieldCount; i++)
                        {
                            testPcehrField[resultSet.GetName(i)] = resultSet.GetString(i);
                        }
                    }
                }
                return testPcehrField;
            }
        }

        /// <summary>
        /// The family name for the test patient.
        /// </summary>
        private string TestPatientFamilyName
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientFamilyName"].ToString();
                }
                else
                {
                    return TestPcehr["FamilyName"];
                }
            }
        }

        /// <summary>
        /// The given name for the test patient.
        /// </summary>
        private string TestPatientGivenName
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientGivenName"].ToString();
                }
                else
                {
                    return TestPcehr["FirstGivenName"];
                }
            }
        }

        /// <summary>
        /// The date of birth for the test patient.
        /// </summary>
        public DateTime TestPatientDateOfBirth
        {
            get
            {
                if (TestDataId == null)
                {
                    return DateTime.Parse(ConfigurationManager.AppSettings["TestPatientDateOfBirth"].ToString());
                }
                else
                {
                    return DateTime.Parse(TestPcehr["DateOfBirth"]);
                }
            }
        }

        /// <summary>
        /// The sex for the test patient.
        /// </summary>
        private SexEnumerator TestPatientSex
        {
            get
            {
                if (TestDataId == null)
                {
                    return (SexEnumerator)Enum.Parse(typeof(SexEnumerator), ConfigurationManager.AppSettings["TestPatientSex"].ToString());
                }
                else
                {
                    switch (TestPcehr["Sex"])
                    {
                        case "M": return SexEnumerator.Male;
                        case "F": return SexEnumerator.Female;
                        case "I": return SexEnumerator.IntersexOrIndeterminate;
                        case "N": return SexEnumerator.NotStatedOrInadequatelyDescribed;
                        default: throw new InvalidOperationException("Invalid Sex in TestPcehr table");
                    }
                }
            }
        }

        /// <summary>
        /// The valid IHI for the test patient.
        /// </summary>
        public string TestPatientValidIhi
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientValidIhi"].ToString();
                }
                else
                {
                    return TestPcehr["Ihi"];
                }
            }
        }

        /// <summary>
        /// An invalid IHI for the test patient.
        /// </summary>
        public string TestPatientInvalidIhi
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientInvalidIhi"].ToString();
                }
                else
                {
                    // Add one to the 15-character prefix of the valid IHI and recalculate the check digit
                    string fifteenDigit = (long.Parse(TestPcehr["Ihi"].Substring(0, 15)) + 1).ToString();
                    return Nehta.VendorLibrary.Common.Validation.CalculateLuhnCheckDigit(fifteenDigit);
                }
            }
        }

        /// <summary>
        /// The hospital to use for CCA testing, specified in the app.config file for the CCA test project.
        /// </summary>
        private static int TestHospitalId
        {
            get
            {
                if (OverrideHospitalId.HasValue)
                {
                    return OverrideHospitalId.Value;
                }
                else
                {
                    return int.Parse(ConfigurationManager.AppSettings["TestHospitalId"].ToString());
                }
            }
        }

        /// <summary>
        /// The Patient Master ID will be inserted into the placeholder in this format to generate an MRN for the test patient.
        /// </summary>
        private static string TestPatientMrnFormat
        {
            get
            {
                return ConfigurationManager.AppSettings["TestPatientMrnFormat"].ToString();
            }
        }

        /// <summary>
        /// The number of days old to make the IHI when it should be outside the validation period, specified in the app.config file for the CCA test project.
        /// </summary>
        private static int OutsideValidationPeriodDays
        {
            get
            {
                return int.Parse(ConfigurationManager.AppSettings["OutsideValidationPeriodDays"].ToString());
            }
        }

        /// <summary>
        /// The Record Access Code (RAC, aka PACC) for the test patient.
        /// </summary>
        public string TestPatientRecordAccessCode
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientRecordAccessCode"].ToString();
                }
                else
                {
                    return TestPcehr["RecordCode"];
                }
            }
        }

        /// <summary>
        /// The Limited Document Access Code (LDAC, aka PACCX) for the test patient.
        /// </summary>
        public string TestPatientLimitedDocumentAccessCode
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientLimitedDocumentAccessCode"].ToString();
                }
                else
                {
                    return TestPcehr["DocumentCode"];
                }
            }
        }

        /// <summary>
        /// The path to a CDA document file to upload.
        /// </summary>
        public static string TestDocumentPath
        {
            get
            {
                return ConfigurationManager.AppSettings["TestDocumentPath"].ToString();
            }
        }

        /// <summary>
        /// The The Patient Master ID will be inserted into the "{0}" placeholder in
        /// this format to generate a source system episode id (visit number) for the
        /// test patient.
        /// </summary>
        public static string TestPatientEpisodeIdFormat
        {
            get
            {
                return ConfigurationManager.AppSettings["TestPatientEpisodeIdFormat"].ToString();
            }
        }

        /// <summary>
        /// Number of seconds to wait for queued operation to change from Pending status.
        /// </summary>
        public static int PcehrQueueTimeout
        {
            get
            {
                return int.Parse(ConfigurationManager.AppSettings["PcehrQueueTimeout"].ToString());
            }
        }

        #endregion Configuration Properties

        #region Test Targets

        // Note, the base class provides the TargetPatient property.

        /// <summary>
        /// The hospital patient to use for testing
        /// </summary>
        public HospitalPatient TargetHospitalPatient
        {
            get
            {
                if (testHospitalPatientField == null)
                {
                    string mrn = string.Format(TestPatientMrnFormat, TargetPatient.PatientMasterId);
                    testHospitalPatientField = InsertHospitalPatient(mrn);
                }
                return testHospitalPatientField;
            }
        }

        /// <summary>
        /// The hospital to use for testing.
        /// </summary>
        public Hospital TargetHospital
        {
            get
            {
                if (testHospitalField == null)
                {
                    ReloadTargetHospital();
                }
                return testHospitalField;
            }
        }

        /// <summary>
        /// The episode to use for testing.
        ///
        /// If the patient identifier type is ValidatedIhi then the episode is not actually inserted into the database by this method.
        /// </summary>
        public Episode TargetEpisode
        {
            get
            {
                if (testEpisodeField == null)
                {
                    testEpisodeField = InsertEpisode();
                }
                return testEpisodeField;
            }
        }

        /// <summary>
        /// The code system of the code to use for this hospital. This value
        /// should exist in CodeSystem.Code. There should be a row in
        /// HospitalCode that links the CodeSystemId of the row in CodeSystem
        /// to the HospitalId that was specified above, and contains a code
        /// for the hospital.
        /// </summary>
        private string HospitalCodeSystem
        {
            get
            {
                return ConfigurationManager.AppSettings["HospitalCodeSystem"].ToString();
            }
        }

        /// <summary>
        /// When looking for the episode that HIPS created for a call with a ValidatedIhi,
        /// it will be matched by admission date/time with a tolerance set here.
        /// </summary>
        private double MaximumAdmissionTimeDifferenceSeconds
        {
            get
            {
                return double.Parse(ConfigurationManager.AppSettings["MaximumAdmissionTimeDifferenceSeconds"].ToString());
            }
        }

        private string targetHospitalCode = null;

        /// <summary>
        /// Gets the hospital code.
        /// </summary>
        private string TargetHospitalCode
        {
            get
            {
                if (targetHospitalCode == null)
                {
                    List<HospitalCode> codes = new HIPS.PcehrDataStore.DataAccess.HospitalCodeDl().GetAll(TestHospitalId);
                    HospitalCode theCode = codes.FirstOrDefault(a => a.CodeSystemCode == HospitalCodeSystem);
                    if (theCode == null)
                    {
                        Assert.Fail("The hospital {0} does not have a code of type {1}", TestHospitalId, HospitalCodeSystem);
                    }
                    targetHospitalCode = theCode.Code;
                }
                return targetHospitalCode;
            }
        }

        /// <summary>
        /// The Medicare number for the test patient.
        /// If 'HasNoMedicareNumber' is true, returns null.
        /// If 'HasInvalidMedicareNumber' is true, increments the first digit modulo 10, to obtain
        /// a number that will not pass the check digit algorithm.
        /// </summary>
        public string TestPatientMedicareNumber
        {
            get
            {
                if (HasNoMedicareNumber)
                {
                    return null;
                }
                string validMedicareNumber;
                if (TestDataId == null)
                {
                    validMedicareNumber = ConfigurationManager.AppSettings["TestPatientMedicareNumber"];
                }
                else
                {
                    validMedicareNumber = TestPcehr["MedicareCardNumber"];
                }

                if (HasInvalidMedicareNumber)
                {
                    int firstDigit = int.Parse(validMedicareNumber.Substring(0, 1));
                    string remainder = validMedicareNumber.Substring(1);
                    firstDigit = (firstDigit + 1) % 10;
                    return string.Format("{0}{1}", firstDigit, remainder);
                }
                return validMedicareNumber;
            }
        }

        /// <summary>
        /// The Medicare IRN for the test patient.
        /// </summary>
        public string TestPatientMedicareIrnNumber
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientMedicareIrnNumber"];
                }
                else
                {
                    return TestPcehr["MedicareIrn"];
                }
            }
        }

        /// <summary>
        /// The DVA or the test patient.
        /// </summary>
        public string TestPatientDVANumber
        {
            get
            {
                if (TestDataId == null)
                {
                    return ConfigurationManager.AppSettings["TestPatientDVANumber"];
                }
                else
                {
                    return TestPcehr["DvaNumber"];
                }
            }
        }


        /// <summary>
        /// Returns a patient identifier that can be sent to HIPS to identify the test patient and hospital.
        /// </summary>
        public PatientIdentifierBase TargetPatientIdentifier
        {
            get
            {
                switch (PatientIdentifierType)
                {
                    case Helpers.PatientIdentifierType.Mrn:
                        return new Mrn(TargetHospitalPatient.Mrn, TargetHospitalCode, HospitalCodeSystem);

                    case Helpers.PatientIdentifierType.PatientMasterId:
                        return new PatientMasterId(TargetPatient.PatientMasterId.Value, null, null, TestHospitalId);

                    case Helpers.PatientIdentifierType.StatePatientId:
                        return new StatePatientId(TargetPatient.StatePatientId, TargetHospitalCode, HospitalCodeSystem);

                    case Helpers.PatientIdentifierType.ValidatedIhi:
                        return new ValidatedIhi(
                            IhiInformation.Ihi,
                            (HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus)IhiInformation.IhiStatusId,
                            (HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus)IhiInformation.IhiRecordStatusId,
                            IhiInformation.DateLastValidated,
                            TestPatientFamilyName,
                            TestPatientGivenName,
                            TestPatientDateOfBirth,
                            TestPatientSex,
                            TargetHospitalCode,
                            HospitalCodeSystem);
                }
                throw new InvalidOperationException();
            }
        }

        /// <summary>
        /// Returns a patient demographic record.
        /// </summary>
        public Demographic TargetDemographic
        {
            get
            {
                return new Demographic(
                    TestPatientFamilyName,
                    TestPatientGivenName,
                    TestPatientDateOfBirth,
                    TestPatientSex,
                    TestPcehr["MedicareCardNumber"],
                    TestPcehr["MedicareIrn"],
                    TargetHospitalCode,
                    HospitalCodeSystem);
            }
        }

        /// <summary>
        /// Obtains the target patient master ID. If the target patient master was inserted
        /// by HIPS rather than by the test suite, then this involves a query on the database.
        /// </summary>
        public int TargetPatientMasterId
        {
            get
            {
                if (!TargetPatient.PatientMasterId.HasValue)
                {
                    GetDatabaseIds(needEpisode: false);
                }
                return TargetPatient.PatientMasterId.Value;
            }
        }


        /// <summary>
        /// Sets a test ParticipatingProvider
        /// </summary>
        public static ParticipatingProvider ParticipatingProvider
        {
            get
            {
                return new ParticipatingProvider()
                {
                    FamilyName = "SANDEEN",
                    GivenNames = "MILO",
                    Hpii = "8003619900000081",
                    //Hpii = null,
                    Title = null,
                    Suffix = null,
                    LocalIdentifier = null
                    //LocalIdentifier = "msand01"
                };
            }
        }

        public PatientMasterIhi IhiInformation
        {
            get;
            set;
        }

        #endregion Test Targets

        #region Public Methods

        /// <summary>
        /// Sets up a patient ready for making a request to the PCEHR. Deletes
        /// the test patient (including all child data, audit records) if it
        /// already exists.
        ///
        /// If the patient identifier type is not ValidatedIhi then the patient,
        /// IHI information and hospital patient will be pre-populated into the
        /// database as if they had come in via the PAS feed.
        ///
        /// However, if the patient identifier type is ValidatedIhi then nothing
        /// will be added to the database, and the IHI information will merely
        /// be stored in the property IhiInformation, for use when you retrieve
        /// the patient identifier using TargetPatientIdentifier.
        /// </summary>
        /// <param name="ihiValid">Whether the IHI is valid.</param>
        /// <param name="validatedWithinPeriod">Whether the IHI was last validated within the configured period.</param>
        /// <param name="hasUnresolvedAlerts">Whether the IHI has unresolved alerts.</param>
        /// <param name="testDataId">Identifies a row in the CcaTestPatients.sdf database.</param>
        /// <param name="preloadPatient">Whether to load the patient into the HIPS database.</param>
        /// <param name="noIhi">Whether to create the patient without an IHI.</param>
        /// <returns>The new CcaPatient object.</returns>
        public static CcaPatient GetPatient(bool ihiValid, bool validatedWithinPeriod, bool hasUnresolvedAlerts, string testDataId = null, bool preloadPatient = true, bool noIhi = false)
        {
            CcaPatient item = new CcaPatient();
            item.TestDataId = testDataId;
            item.ResetPatientData();
            item.IhiInformation = new PatientMasterIhi();
            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                if (preloadPatient)
                {
                    item.IhiInformation.PatientMasterId = item.SetTestData();
                }
            }

            if (ihiValid)
            {
                item.IhiInformation.Ihi = item.TestPatientValidIhi;
                item.IhiInformation.IhiStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus.Active;
                item.IhiInformation.IhiRecordStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus.Verified;
            }
            else
            {
                item.IhiInformation.Ihi = item.TestPatientInvalidIhi;
                item.IhiInformation.IhiStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus.Active;
                item.IhiInformation.IhiRecordStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus.Unverified;
            }

            if (noIhi)
            {
                item.IhiInformation.Ihi = null;
                item.IhiInformation.IhiStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus.Unknown;
                item.IhiInformation.IhiRecordStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus.Unknown;
            }

            if (hasUnresolvedAlerts)
            {
                item.IhiInformation.IhiStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus.DuplicateIhi;
            }

            item.IhiInformation.RegisteredFamilyName = item.TargetPatient.CurrentName.FamilyName;
            item.IhiInformation.RegisteredGivenName = item.TargetPatient.CurrentName.GivenNames;
            item.IhiInformation.RegisteredSexId = item.TargetPatient.CurrentSexId;
            item.IhiInformation.HealthProviderOrganisationNetworkId = item.TargetHospital.HealthProviderOrganisationNetworkId;
            if (validatedWithinPeriod)
            {
                item.IhiInformation.DateLastValidated = DateTime.Now;
            }
            else
            {
                item.IhiInformation.DateLastValidated = DateTime.Now - TimeSpan.FromDays(OutsideValidationPeriodDays);
            }

            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                if (preloadPatient)
                {
                    item.PatientMasterIhiInsert(item.IhiInformation);
                    HospitalPatient hospitalPatient = item.TargetHospitalPatient;
                }
            }
            return item;
        }

        /// <summary>
        /// Gets a test patient without an IHI.
        /// </summary>
        /// <param name="testDataId">The test data ID from the SQL compact database (or null for default)</param>
        /// <returns>The CCA patient</returns>
        public static CcaPatient GetPatientWithoutIhi(string testDataId = null)
        {
            CcaPatient item = new CcaPatient();
            item.TestDataId = testDataId;
            item.ResetPatientData();
            item.IhiInformation = new PatientMasterIhi();
            item.IhiInformation.IhiStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus.Unknown;
            item.IhiInformation.IhiRecordStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus.Unknown;
            item.IhiInformation.DateLastValidated = DateTime.Now;
            item.HasNoMedicareNumber = true;
            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                item.SetTestData();
                HospitalPatient hospitalPatient = item.TargetHospitalPatient;
            }
            return item;
        }

        /// <summary>
        /// Overrides the patient name and if the patient identifier type is not ValidatedIhi
        /// then reinserts the test patient. This method is designed to be used immediately
        /// after GetPatient and before using the patient to make any calls to HIPS.
        /// This should be used to generate a deliberate demographic mismatch.
        /// </summary>
        /// <param name="givenNames">The new given names for the patient</param>
        /// <param name="familyName">The new family name for the patient</param>
        public void OverridePatientName(string givenNames, string familyName)
        {
            TestPcehr["FirstGivenName"] = givenNames;
            TestPcehr["FamilyName"] = familyName;
            TargetPatient.SetNewCurrentName(null, givenNames, familyName, null);
            IhiInformation.RegisteredFamilyName = TargetPatient.CurrentName.FamilyName;
            IhiInformation.RegisteredGivenName = TargetPatient.CurrentName.GivenNames;
            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                IhiInformation.PatientMasterId = SetTestData();
                testHospitalPatientField = null;
                HospitalPatient hp = TargetHospitalPatient;
                PatientMasterIhiInsert(IhiInformation);
            }
        }

        /// <summary>
        /// Resets the date last validated for the IHI to be outside the validation period.
        /// </summary>
        public void MakeIhiStale()
        {
            IhiInformation.DateLastValidated = DateTime.Now - TimeSpan.FromDays(OutsideValidationPeriodDays);

            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                PatientMasterIhiDelete(IhiInformation);
                PatientMasterIhiInsert(IhiInformation);
            }
        }

        /// <summary>
        /// Resets the IHI to be valid.
        /// </summary>
        public void MakeIhiValid()
        {
            IhiInformation.Ihi = TestPatientValidIhi;
            IhiInformation.IhiRecordStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus.Verified;

            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                PatientMasterIhiDelete(IhiInformation);
                PatientMasterIhiInsert(IhiInformation);
            }
        }

        /// <summary>
        /// Resets the IHI to be invalid.
        /// </summary>
        public void MakeIhiInvalid()
        {
            IhiInformation.Ihi = TestPatientInvalidIhi;
            IhiInformation.IhiRecordStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus.Unknown;

            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                PatientMasterIhiDelete(IhiInformation);
                PatientMasterIhiInsert(IhiInformation);
            }
        }

        /// <summary>
        /// Reads the most recent document whose name starts with the specified
        /// document type from the configured directory (TestDocumentPath).
        /// Sets the IHI, increments the ID value and saves a copy of the
        /// updated document under a new name.
        ///
        /// The document ID may be an OID or a UUID. If the document ID is an
        /// OID, this implementation enforces that the last part of the OID in
        /// the id element's root attribute is the same as the extension
        /// attribute, and the value is incremented numerically. If the document
        /// ID is a UUID, then a new UUID is generated for each document.
        /// 
        /// For example, if docType is SampleDocumentType.DischargeSummary then
        /// the initial sample document of that type should have the name
        /// "DischargeSummary.xml". Each file that is written out will have the
        /// form DischargeSummary_NNNN.xml where NNNN is the extension of the
        /// document ID, or the whole document ID if it is a UUID.
        /// </summary>
        /// <returns>The new document</returns>
        public CdaDocument CreateNewDocument(SampleDocumentType docType)
        {
            return CreateDocument(docType, null);
        }

        public CdaDocument CreateAmendedDocument(CdaDocument parent)
        {
            return CreateDocument(parent.GetDocType(), parent);
        }

        private CdaDocument CreateDocument(SampleDocumentType docType, CdaDocument parent)
        {
            DirectoryInfo directory = new FileInfo(TestDocumentPath).Directory;
            string searchPattern = string.Format("{0}*.xml", docType);
            FileInfo lastFile = directory.GetFiles(searchPattern).OrderBy(a => a.CreationTime).Last();

            CdaDocument cdaDoc = new CdaDocument(lastFile);
            cdaDoc.SetIhi(IhiInformation.Ihi);
            string extension = cdaDoc.IncrementDocumentId();
            cdaDoc.SetParentDocument(parent);
            cdaDoc.SetNewObjectIds();
            cdaDoc.SetDateOfBirth(TestPatientDateOfBirth);
            cdaDoc.SetSex(TestPatientSex);
            cdaDoc.SetPatientName(TestPatientFamilyName, TestPatientGivenName);
            cdaDoc.SetCreationTime(DateTime.Now.ToString("yyyyMMddHHmmzzz").Replace(":", ""));
            
            // Save the amended document
            string[] nameParts = lastFile.Name.Split('.', '_');
            string newFullName = string.Format("{0}\\{1}_{2}.xml", directory.FullName, nameParts[0], extension);
            cdaDoc.Save(newFullName);

            return cdaDoc;
        }

        public UploadDischargeSummaryLevel1ARequest CreateDocument1ARequest()
        {

            Episode episode = this.TargetEpisode;

            byte[] pdf = this.GetPDF("1A_Sample_Upload");


            DateTime admission = this.TargetEpisode.AdmissionDate;
            LocalUser user = this.GetDTOLocalTestUser();

            ParticipatingProvider provider = CcaPatient.ParticipatingProvider;

            //ParticipatingProvider provider = new ParticipatingProvider()
            //{
            //    FamilyName = "SANDEEN",
            //    GivenNames = "MILO",
            //    Hpii = "8003619900000081",
            //    Title = null,
            //    Suffix = null,
            //    LocalIdentifier = null
            //};

            HIPS.ServiceContracts.Common.DTO.PatientIdentifier.Mrn patientIdentifier = new HIPS.ServiceContracts.Common.DTO.PatientIdentifier.Mrn()
            {
                Value = this.TargetHospitalPatient.Mrn,
                HospitalCode = this.TargetPatientIdentifier.HospitalCode,
                HospitalCodeSystem = this.TargetPatientIdentifier.HospitalCodeSystem
            };

            CdaHeaderMetadata metadata = new CdaHeaderMetadata()
            {
                AdmissionDateTime = admission,
                DischargeDateTime = admission.AddDays(1),
                DocumentAuthor = provider,
                DocumentCreationDateTime = DateTime.Now,
                LegalAuthenticator = provider,
                ModeOfSeparation = ModeOfSeparation.Home,
                ResponsibleHealthProfessional = provider,
                SourceDocumentStatus = SourceDocumentStatus.Interim,
                Specialty = "General Practitioner"
            };

            CdaAttachment[] attachments = new CdaAttachment[0];

            UploadDischargeSummaryLevel1ARequest request = new UploadDischargeSummaryLevel1ARequest
            {
                User = user,
                PdfDocument = pdf,
                PatientIdentifier = patientIdentifier,
                CdaHeaderMetadata = metadata,
                Attachments = attachments.ToList()
            };

            return request;

        }

        public byte[] GetPDF(string name)
        {
            DirectoryInfo directory = new FileInfo(TestDocumentPath).Directory;
            string searchPattern = string.Format("{0}*.pdf", name);
            FileInfo file = directory.GetFiles(searchPattern).OrderBy(a => a.CreationTime).Last();

            return File.ReadAllBytes(file.FullName);
        }
        /// <summary>
        /// Gets all the downloaded documents for the target patient.
        /// </summary>
        /// <returns>A list of downloaded document.</returns>
        public List<DownloadedDocument> GetAllDownloadedDocuments()
        {
            if (!TargetPatient.PatientMasterId.HasValue)
            {
                GetDatabaseIds(needEpisode: false);
            }
            DownloadedDocumentDl dataAccess = new DownloadedDocumentDl(GetTestUser());
            return dataAccess.GetAll(TargetPatient.PatientMasterId.Value);
        }

        /// <summary>
        /// Gets the PCEHR message queue items for the target episode.
        /// </summary>
        /// <param name="episodeId">The episode ID</param>
        /// <returns></returns>
        public List<PcehrMessageQueue> GetPcehrMessageQueueItems()
        {
            if (!TargetEpisode.EpisodeId.HasValue)
            {
                GetDatabaseIds(needEpisode: true);
            }
            GenericDataAccess dataAccess = new GenericDataAccess();
            List<PcehrMessageQueue> result;
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT * from [hips].[PcehrMessageQueue] WHERE EpisodeId = {0} ORDER BY PcehrMessageQueueId DESC", TargetEpisode.EpisodeId);
                result = GetPopulatedBusinessList<PcehrMessageQueue>(command.ExecuteReader());
            }
            return result;
        }

        public void ReloadTargetHospital()
        {
            testHospitalField = new HospitalDl().Get(TestHospitalId);
        }

        #endregion Public Methods

        #region Private Implementation

        private HospitalPatient testHospitalPatientField;
        private Episode testEpisodeField;
        private Hospital testHospitalField;

        /// <summary>
        /// The constructor is private. Use GetPatient to allocate an instance of CcaPatient.
        /// </summary>
        private CcaPatient()
        {
        }

        /// <summary>
        /// Returns a SQL command to match the target patient in the database,
        /// so that the test suite can remove all records relating to that
        /// patient, and start each test from a clean state.
        /// </summary>
        /// <returns></returns>
        public override string SqlCommandText()
        {
            return string.Format(
                @"SELECT DISTINCT pm.PatientMasterId FROM hips.PatientMaster pm
                INNER JOIN hips.PatientMasterName pmn on pm.PatientMasterId = pmn.PatientMasterId
                WHERE DateOfBirth = '{0:yyyy-MM-dd}'
                AND FamilyName = '{1}'
                AND GivenNames = '{2}'
                ORDER BY pm.PatientMasterId DESC",
                TestPatientDateOfBirth,
                TestPatientFamilyName,
                TestPatientGivenName);
        }

        /// <summary>
        /// When calling HIPS with a ValidatedIhi, the database primary keys for the
        /// patient master, hospital patient and episode are allocated by HIPS and
        /// not by this test suite. Therefore we will query to find them from the
        /// database, using the same query as was used to find and remove the patient.
        /// </summary>
        internal void GetDatabaseIds(bool needEpisode)
        {
            // Get the Patient Master ID
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = SqlCommandText();
                int? result = System.Convert.ToInt32(command.ExecuteScalar());
                if (result.HasValue && result > 0)
                {
                    TargetPatient.PatientMasterId = result;
                }
                command.Connection.Close();
            }
            if (!TargetPatient.PatientMasterId.HasValue)
            {
                Assert.Fail(DialogueResource.PatientNotFound);
            }

            // Get the Hospital Patient and Episode
            UserDetails user = GetTestUser();
            HospitalPatient hospitalPatient = new HospitalPatientDl(user).GetAll(TestHospitalId, TargetPatient.PatientMasterId.Value).FirstOrDefault();

            //null occurs when not reloading the patient and a hospital record is not created
            if (hospitalPatient != null)
            {
                //All Episode LifeCycle Types can be used in this case
                var matched = from episode in new EpisodeDl(user).GetAll(hospitalPatient.PatientId.Value, null)
                              where Math.Abs((episode.AdmissionDate - TargetEpisode.AdmissionDate).TotalMinutes) < MaximumAdmissionTimeDifferenceSeconds
                              select episode;
                if (matched.Count() == 1)
                {
                    Episode episode = matched.First();
                    TargetEpisode.EpisodeId = episode.EpisodeId.Value;
                }
                else if (needEpisode)
                {
                    Assert.Fail(DialogueResource.EpisodeNotFound);
                }
                TargetHospitalPatient.PatientId = hospitalPatient.PatientId.Value;
            }
            else
            {
                TargetHospitalPatient.PatientId = null;
                //if an episode is needed then this needs to fail
                if (needEpisode)
                {
                    Assert.Fail(DialogueResource.EpisodeNotFound);
                }
            }
            
        }

        /// <summary>
        /// Overrides the basic patient with the one we want to use for CCA testing.
        /// </summary>
        override public void GetTestInsertPatient()
        {
            base.GetTestInsertPatient();
            TargetPatient.Names = new List<PatientMasterName>();
            TargetPatient.SetNewCurrentName(null, TestPatientGivenName, TestPatientFamilyName, null);
            TargetPatient.CurrentSexId = (int)TestPatientSex;
            TargetPatient.DateOfBirth = TestPatientDateOfBirth;
            TargetPatient.DvaNumber = TestPatientDVANumber;
            TargetPatient.MedicareNumber = TestPatientMedicareNumber;
            TargetPatient.MedicareIrn = TestPatientMedicareIrnNumber;
        }

        /// <summary>
        /// Creates an episode for the test patient and returns the episode that was created.
        ///
        /// If the patient identifier type is ValidatedIhi then the episode is not actually inserted into the database by this method.
        /// </summary>
        /// <returns>The episode that was created</returns>
        private Episode InsertEpisode()
        {
            Episode item = new Episode();
            
            item.AdmissionDate = DateTime.Now;
            item.SourceSystemEpisodeId = string.Format(TestPatientEpisodeIdFormat, TargetHospitalPatient.PatientMasterId);
            item.ResponsibleProviderId = -1;
            item.EpisodeLifecycleId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.EpisodeLifecycle.Admitted;
            item.EpisodeTypeId = -1;

            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi && TargetHospitalPatient.PatientId != null)
            {
                item.PatientId = TargetHospitalPatient.PatientId.Value;
                GenericDataAccess dataAccess = new GenericDataAccess();
                using (SqlCommand command = TestCommand)
                {
                    command.CommandText = "[hips].[EpisodeInsert]";
                    command.CommandType = System.Data.CommandType.StoredProcedure;
                    Insert<Episode>(item, command);
                    command.CommandType = System.Data.CommandType.Text;
                }
            }
            return item;
        }

        /// <summary>
        /// Creates a hospital patient record with the specified MRN.
        /// If the patient identifier type is Validated IHI then the record
        /// will not be saved to the database.
        /// </summary>
        /// <param name="item">The item.</param>
        private HospitalPatient InsertHospitalPatient(string mrn)
        {
            HospitalPatient item = new HospitalPatient();
            item.HospitalId = TestHospitalId;
            item.Mrn = mrn;

            if (PatientIdentifierType != Helpers.PatientIdentifierType.ValidatedIhi)
            {
                item.PatientMasterId = TargetPatient.PatientMasterId.Value;
                GenericDataAccess dataAccess = new GenericDataAccess();
                using (SqlCommand command = TestCommand)
                {
                    command.CommandText = "[hips].[HospitalPatientInsert]";
                    command.CommandType = System.Data.CommandType.StoredProcedure;
                    Insert<HospitalPatient>(item, command);
                    command.CommandType = System.Data.CommandType.Text;
                }
            }
            return item;
        }

        /// <summary>
        /// Gets all the IHI lookup audit records for the target patient.
        /// </summary>
        /// <returns></returns>
        internal List<IhiLookupAudit> GetIhiLookupAudits()
        {
            int patientMasterId = TargetPatientMasterId;
            GenericDataAccess dataAccess = new GenericDataAccess();
            List<IhiLookupAudit> result;
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT * from [hips].[IhiLookupAudit] WHERE PatientMasterId = {0} ORDER BY IhiLookupAuditId", patientMasterId);
                result = GetPopulatedBusinessList<IhiLookupAudit>(command.ExecuteReader());
            }
            return result;
        }

        /// <summary>
        /// Gets all the PCEHR audit records for the target patient.
        /// </summary>
        /// <returns></returns>
        internal List<PcehrAudit> GetPcehrAudits()
        {
            int patientMasterId = TargetPatientMasterId;
            GenericDataAccess dataAccess = new GenericDataAccess();
            List<PcehrAudit> result;
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT * from [hips].[PcehrAudit] WHERE PatientMasterID = {0} ORDER BY PcehrAuditId", patientMasterId);
                result = GetPopulatedBusinessList<PcehrAudit>(command.ExecuteReader());
            }
            return result;
        }

        /// <summary>
        /// Gets all the consent audit records for the target episode.
        /// </summary>
        /// <returns></returns>
        internal List<ConsentAudit> GetConsentAudits()
        {
            if (!TargetEpisode.EpisodeId.HasValue)
            {
                GetDatabaseIds(needEpisode: true);
            }
            int episodeId = TargetEpisode.EpisodeId.Value;
            GenericDataAccess dataAccess = new GenericDataAccess();
            List<ConsentAudit> result;
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT * from [hips].[ConsentAudit] WHERE EpisodeId = {0} ORDER BY ConsentAuditId", episodeId);
                result = GetPopulatedBusinessList<ConsentAudit>(command.ExecuteReader());
            }
            return result;
        }

        /// <summary>
        /// Gets all the contact details for the target patient.
        /// </summary>
        /// <returns></returns>
        public List<Contact> GetCurrentContacts()
        {
            ContactDl dataAccess = new ContactDl();
            return dataAccess.GetAllByPatient(TargetPatientMasterId);
        }

        /// <summary>
        /// Gets the signing certificate from the Windows certificate repository.
        /// </summary>
        /// <returns>Matching certificate.</returns>
        public X509Certificate2 GetSigningCertificate()
        {
            StoreLocation storeLocation = StoreLocation.LocalMachine;
            StoreName storeName = StoreName.My;
            X509FindType findType = X509FindType.FindBySerialNumber;
            HospitalDl hospitalDl = new HospitalDl();
            string findValue = hospitalDl.Get(TargetHospitalPatient.HospitalId).PcehrCertSerial;
            bool valid = false;

            X509Store certStore = new X509Store(storeName, storeLocation);
            certStore.Open(OpenFlags.ReadOnly);

            X509Certificate2Collection foundCerts =
              certStore.Certificates.Find(findType, findValue, valid);
            certStore.Close();

            // Check if any certificates were found with the criteria
            if (foundCerts.Count == 0)
                throw new ArgumentException("Certificate was not found with criteria '" + findValue + "'");

            // Check if more than one certificate was found with the criteria
            if (foundCerts.Count > 1)
                throw new ArgumentException("More than one certificate found with criteria '" + findValue + "'");

            return foundCerts[0];
        }

        #endregion Private Implementation
    }
}