﻿using System;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_301
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems
    ///                                 Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.301
    /// Use Case Name:	Access a View Service
    /// Test Case ID:   PCEHR_CIS_019119
    /// Objective:
    /// To check that the Clinical Information System provides a warning to the
    /// CIS User if the healthcare recipient's demographic information in a
    /// clinical document downloaded from the PCEHR System does not match the
    /// demographic information in the local healthcare recipient's record.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019119 : CcaTest
    {
        /// <summary>
        /// For a healthcare recipient whose PCEHR contains a clinical document,
        /// edit the healthcare recipient's demographic information locally to 
        /// create a mismatch between the demographics details contained within
        /// the clinical document on the PCEHR and the local demographic details.
        /// 
        /// Perform an operation to request a view of the PCEHR that is a clinical
        /// document for the healthcare recipient whose demographic details were
        /// edited locally to simulate a mismatch:
        /// 
        /// a. Verify that a warning is displayed to the user alerting them to a 
        /// mismatch between the local demographic information and the demographic
        /// information contained in the clinical document obtained from the PCEHR
        /// for the healthcare recipient.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_301")]
        public void CIS_301_019119_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            
            // Edit the demographic information to create a mismatch
            patient.OverridePatientName("Someone", "Else");
            
            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, "GainAccessEmergency");

            PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
            {
                FromDate = DateTime.Parse("1/1/2000"),
                ToDate = DateTime.Now
            };
            ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
            LogAssert.ExpectResponse(HipsResponseIndicator.DemographicMismatchWarning, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);

            CdaDocument doc = new CdaDocument(viewResponse.Document);
            bool dobMatched = patient.TargetPatient.DateOfBirth.Date == doc.PatientDateOfBirth.Date;
            bool sexMatched = patient.TargetPatient.CurrentSexId == (int)doc.PatientSex;
            bool familyNameMatched = false;
            foreach (PatientMasterName docName in doc.PatientNames)
            {
                PatientMasterName localName = patient.TargetPatient.CurrentName;
                if (docName.FamilyName.ToUpper() == localName.FamilyName.ToUpper())
                {
                    familyNameMatched = true;
                    break;
                }
            }
            Log("Document DOB {0:dd/MM/yyyy} {1} match local DOB {2:dd/MM/yyyy}", doc.PatientDateOfBirth, dobMatched ? "did" : "did not", patient.TargetPatient.DateOfBirth);
            Log("Document sex {0} {1} match local sex {2}", doc.PatientSex, sexMatched ? "did" : "did not", (SexEnumerator)patient.TargetPatient.CurrentSexId);
            Log("Document family name {0} {1} match local family name {2}", doc.PatientNames[0].FamilyName, familyNameMatched ? "did" : "did not", patient.TargetPatient.CurrentName.FamilyName);
            LogAssert.IsTrue(!sexMatched || !dobMatched || !familyNameMatched, "There was at least one demographic mismatch", "There were no demographic mismatches");
        }
    }
}