﻿using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using Test.CommonCcaNoc.Helpers;
using Test.PcehrCcaNoc.NoticeOfConnection.DocumentConsumption;
using Test.PcehrCcaNoc.NoticeOfConnection.DocumentProduction;
using Test.PcehrCcaNoc.NoticeOfConnection.Registration;

namespace Test.PcehrCcaNoc.NoticeOfConnection.ClientTypeChange
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway - Change of Client System Type Test Plan
    ///                                 Version 4.4 - 19 January 2015
    ///                                 
    /// This test plan is only to be used when a Software Vendor's product has
    /// already completed the initial NOC Test Plan and has developed the
    /// ability to act as a different client system type. The services tested
    /// on this test plan must match those already passed as the previous 
    /// client system type. If a service type has not been tested previously,
    /// a full set of test cases must be executed as per the standard test
    /// plan. For example - Initially tested as a CIS, but has added CSP
    /// functionality for services already tested and passed.
    /// 
    /// Be sure to set the HospitalId in the app.config to a hospital that
    /// has CSP enabled.
    /// </summary>
    [TestClass]
    public class NOC_ClientTypeChange : CcaTest
    {
        /// <summary>
        /// Operations: getChangeHistoryView
        /// Test No: ClientTypeChange - 1
        /// Related Test Plan Test: 6
        /// Conformance Points: VIEW-L8 VIEW-L9 VIEW-L10 VIEW-T8 VIEW-T11 
        /// Test Type: Positive
        /// Role: Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input: 
        ///     Ensure the Document UUID is used for the document unique identifier
        ///     which is requested for the document change history view operation.
        /// Expected Result:
        ///     The request is transmitted successfully.
        ///     Confirmation that the payload is well formed and complies with the getChangeHistoryView Request XML Schema. 
        ///     The document id  is provided.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_01()
        {
            this.RunTest<NOC_GetChangeHistoryView>(test => test.PCEHR_NOC_006());
        }

        /// <summary>
        /// Operations	
        ///     getDocumentList
        /// Test No.
        ///     ClientTypeChange - 3
        /// Related Test Plan Test No.
        ///     14
        /// Conformance Points
        ///     VIEW-L20 VIEW-L21 VIEW-L22 VIEW-L58  VIEW-T15 VIEW-T16 VIEW-T17 VIEW-T18 VIEW-T19
        ///     DEXS-L24 DEXS-L25 DEXS-L26 DEXS-L27 DEXS-L133  DEXS-T23 DEXS-T24 DEXS-T26
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a getDocumentList query to retrieve a list of approved documents.
        ///     Ensure Patient Id that needs to be used to find documents is in the correct format as shown in the following example: 
        ///     {0}^^^&1.2.36.1.2001.1003.0&ISO,
        ///     where {0} – is the 16 digit HI number
        ///     The query id parameter for the getDocumentList query is set to urn:uuid:14d4debf-8f97-4251-9a74-a90016b0af0d 
        ///     $XDSDocumentEntryStatus is set to urn:oasis:names:tc:ebxml-regrep:StatusType:Approved 
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR System.
        ///     Confirmation that the payload is well formed and complies with the B2B DocumentRegistry interface.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_03()
        {
            this.RunTest<NOC_GetDocumentList>(test => test.PCEHR_NOC_014());
        }

        /// <summary>
        /// Operations	
        ///     For All View Service Operations
        /// Test No.
        ///     ClientTypeChange - 4
        /// Related Test Plan Test No.
        ///     17
        /// Conformance Point
        ///     VIEW-L52 VIEW-L53 VIEW-L54  VIEW-T1 VIEW-T2 VIEW-T3 VIEW-T4 VIEW-T20 VIEW-T31 VIEW-T32 VIEW-T34
        ///     VIEW-T36 VIEW-T38 VIEW-T39 VIEW-T40 VIEW-T47 VIEW-T48  Common Header VIEW-L55 VIEW-L56 VIEW-L57
        ///     VIEW-L58 VIEW-L59 VIEW-L60 VIEW-L61 VIEW-L63 VIEW-L64 VIEW-L65 VIEW-L66 VIEW-L67 VIEW-L68 VIEW-L69
        ///     VIEW-L70 VIEW-L71 VIEW-L72 VIEW-L73 VIEW-L74 VIEW-L75 VIEW-L76 VIEW-L77
        ///     VIEW-T22 VIEW-T23 VIEW-T24 VIEW-T25 VIEW-T26 VIEW-T27 VIEW-T28 VIEW-T29
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Ensure the client system includes PCEHR Common Header Elements and
        ///     Transmission Signature containing a signed attestation of elements
        ///     contained within the SOAP message on all SOAP Requests.
        ///     - For this test case, the client system uses valid HPI-O certificates
        ///       provided in the test data package. Service Invoker and Service Provider
        ///       SHALL establish a secure communication channel when interacting with each other.
        ///     - Implementation confirms to web services base profile and TLS security profile.
        ///     - Unique message ID  is used in all transactions.
        ///     - IHI number is specified in all transactions, is valid, matches the individual
        ///       who owns the PCEHR and is numerical.
        ///     - ID is specified in all transactions, has no leading or trailing spaces. 
        ///     - If ID type is HPII, the ID specified is valid and numerical.
        ///     - If ID type is LocalSystemIdentifier, the ID specified is a representation of
        ///       the access credential used to access the system originating the request.
        ///     - If role and/or alternate organisation is supplied, no leading or trailing spaces used                     
        ///     - User name, vendor, product name, production version, platform, organisation ID,
        ///       organisation name,  has no leading or trailing space, it is not blank.
        ///     - Organisation name is set to the name of the organisation.
        ///     - Product name, product version, product type and product platform are set to the
        ///       appropriate values as per client system.
        /// Expected Result
        ///     The requests are transmitted successfully to the PCEHR System. Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_04()
        {
            this.RunTest<NOC_AllViewServiceOperations>(test => test.PCEHR_NOC_017());
        }

        /// <summary>
        /// Operations
        ///     doesPCEHRExist
        /// Test No.
        ///     ClientTypeChange - 5
        /// Related Test Plan Test No.
        ///     18
        /// Conformance Point
        ///     RCAS-L1 RCAS-L2 RCAS-L3 RCAS-L4 RCAS-L5 RCAS-L6 
        ///     RCAS-L40 (CIS)
        ///     RCAS-L49 (CSP)
        ///     RCAS-T5 RCAS-T7 RCAS-T8 RCAS-T9 RCAS-T10
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Ensure a verified IHI Number format for the individual who owns the
        ///     PCEHR is used to perform the operation with a unique message ID. 
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR System.
        ///     Confirmation that the payload is well formed and complies with
        ///     doesPCEHRExist Request XML Schema
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_05()
        {
            this.RunTest<NOC_DoesPcehrExist>(test => test.PCEHR_NOC_018());
        }

        /// <summary>
        /// Operations
        ///     gainPCEHRAccess
        /// Test No.
        ///     ClientTypeChange - 6
        /// Related Test Plan Test No.
        ///     22
        /// Conformance Point
        ///     RCAS-L7 RCAS-L16 RCAS-L17 RCAS-L19 RCAS-L20 RCAS-L21
        ///     RCAS-L41 (CIS) 
        ///     RCAS-L50 (CSP)
        ///     RCAS-T16 RCAS-T17 RCAS-T18 RCAS-T19 RCAS-T21
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Ensure a valid IHI number format and healthcare provider UserID are used to gain access to a PCEHR.
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR system.
        ///     Confirmation that the payload is well formed and complies with the gainPCEHRAccess Request XML schema. 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_06()
        {
            this.RunTest<NOC_GainPcehrAccess>(test => test.PCEHR_NOC_022());
        }

        /// <summary>
        /// Operations
        ///     For All Record Access Services
        /// Test No.
        ///     ClientTypeChange - 7
        /// Related Test Plan Test No.
        ///     32
        /// Conformance Point
        ///     RCAS-L25 RCAS-L26 RCAS-L27 RCAS-L28 RCAS-L29 RCAS-L31
        ///     RCAS-L35 (CIS) RCAS-L36 (CIS) RCAS-L38 (CIS) RCAS-L39 (CIS) RCAS-L40 (CIS) RCAS-L41 (CIS)
        ///     RCAS-L42 (CSP) RCAS-L43 (CSP) RCAS-L44 (CSP) RCAS-L45 (CSP) RCAS-L48 (CSP) RCAS-L49 (CSP) RCAS-L50 (CSP)
        ///     RCAS-L58 RCAS-L59 RCAS-L60 RCAS-L61 RCAS-L62 RCAS-L63 RCAS-L64 RCAS-L65 RCAS-L66 RCAS-L67
        ///     RCAS-L68 RCAS-L69 RCAS-L70 RCAS-L71 RCAS-L72 RCAS-L73 RCAS-L74 RCAS-L75 RCAS-L76 
        ///     RCAS-T1 RCAS-T2 RCAS-T3 RCAS-T28 RCAS-T30 RCAS-T31 RCAS-T32 RCAS-T33 RCAS-T34 RCAS-T35 RCAS-T36 RCAS-T37
        ///     RCAS-T38 (CIS)
        ///     RCAS-T39 (CSP)
        ///     RCAS-T42 RCAS-T43 RCAS-T45 RCAS-T46 RCAS-T47 RCAS-T48 RCAS-T49 RCAS-T50
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Ensure the client system includes PCEHR Common Header Elements
        ///     and Transmission Signature containing a signed attestation of 
        ///     elements contained within the SOAP message on all SOAP Requests.
        ///     - For this test case, the client system uses valid HPI-O
        ///       certificates provided in the test data package. Service
        ///       Invoker and Service Provider SHALL establish a secure
        ///       communication channel when interacting with each other.
        ///     - Implementation confirms to web services base profile and TLS
        ///       security profile.
        ///     - Unique message ID  is used in all transactions.
        ///     - IHI number is specified in all transactions, is valid, matches
        ///       the individual who owns the PCEHR and is numerical.
        ///     - ID is specified in all transactions, has no leading or
        ///       trailing spaces. 
        ///     - If ID type is HPII, the ID specified is valid and numerical.
        ///     - If ID type is LocalSystemIdentifier, the ID specified is a
        ///       representation of the access credential used to access the
        ///       system originating the request.
        ///     - If role and/or alternate organisation is supplied, no leading
        ///       or trailing spaces used                     
        ///     - User name, vendor, product name, production version, platform,
        ///       organisation ID, organisation name, has no leading or trailing
        ///       space, it is not blank.
        ///     - Organisation name is set to the name of the organisation.
        ///     - Product name, product version, product type and product
        ///       platform are set to the appropriate values as per client system.
        ///     - Service Invoker and Service Provider System SHALL comply with
        ///       audit requirements as per section 5.3 (Data Handling) in the
        ///       National E-Health Security and Access Framework [NeSAF].
        /// Expected Result
        ///     The requests are transmitted successfully to the PCEHR System.
        ///     Expected response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_07()
        {
            this.RunTest<NOC_AllRecordAccessServices>(test => test.PCEHR_NOC_032());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 8
        /// Related Test Plan Test No.
        ///     33
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154
        ///     DEXS-L155 DEXS-L156 DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117
        ///     DEXS-L118 DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 DEXS-T55
        ///     DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 DEXS-T62
        ///     DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T133 DEXS-T138 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.03 UPLOAD.04 
        ///     CLD.06 CLD.07 CLD.08 CLD.09 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Shared Health Summary document. 
        /// Expected Result
        ///     - The document is submitted successfully with the input data
        ///       field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange
        ///       Common Header
        ///     - The Provide & Register Document Set transactions contains
        ///       exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains
        ///       exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT
        ///       contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder
        ///       Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///       Field Name of Table 2 in the PCEHR Document Exchange Service
        ///       TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI
        ///       number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID
        ///       field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the
        ///       XDSDocumentEntry.patientId field within the supplied CDA
        ///       Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template
        ///       ID of the Template to which the CDA Document asserts
        ///       conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed
        ///       value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///       Field Name of Table 3 in the  PCEHR Document Exchange Service
        ///       TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier
        ///       of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the 
        ///       value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to
        ///       the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the
        ///       value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///       identical with the value provided within the
        ///       XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the
        ///       value provided within the XDSDocumentEntry.sourcePatientId
        ///       entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to
        ///       'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been
        ///       converted to UTC if the time within the CDA document contains
        ///       a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in
        ///       the same folder as the CDA_ROOT.xml for the purpose of
        ///       sending to the PCEHR system
        ///     - The XDSDocumentEntry.authorInstitution name field matches the
        ///       Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author
        ///       name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode
        ///       fields are set to  '60591-5'
        ///     - The XDSDocumentEntry.classCodeDisplayName and 
        ///       XDSDocumentEntry.typeCodeDisplayName are set to
        ///       'Shared Health Summary'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA
        ///      effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the Document
        ///       Creation Time (XDSDocumentEntry.creationTime)
        ///     - The XDSDocumentEntry.serviceStopTime is set to the Document
        ///       Creation Time (XDSDocumentEntry.creationTime)
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)- 
        ///     - The Title slot is present and set to 'Shared Health Summary'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)- 
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///       as-is if the time within the CDA document contains a 
        ///       date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not
        ///       padded with zeroes to increase precision if the time within
        ///       the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_08()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_033());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 9
        /// Related Test Plan Test No.
        ///     34
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154
        ///     DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117
        ///     DEXS-L118 DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 DEXS-T55
        ///     DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 DEXS-T62
        ///     DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T133 DEXS-T138 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.07 UPLOAD.08
        ///     CLD.06 CLD.07 CLD.12 CLD.13 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Event Summary document. 
        /// Expected Result
        ///     - The document is submitted successfully with the input data
        ///       field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     -  Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     -  WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     -  Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the 
        ///       XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template
        ///       ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///       Field Name of Table 3 in the  PCEHR Document Exchange Service
        ///       TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the
        ///       value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to
        ///       the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the
        ///       value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///       identical with the value provided within the 
        ///       XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the
        ///       value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been
        ///       converted to UTC if the time within the CDA document contains
        ///       a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in
        ///       the same folder as the CDA_ROOT.xml for the purpose of
        ///       sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '34133-9'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Event Summary'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the encounter start date-time (encompassingEncounter/effectiveTime)
        ///     - The XDSDocumentEntry.serviceStopTime is set to the encounter end date-time (encompassingEncounter/effectiveTime)
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)
        ///     - The Title slot is present and set to 'Event Summary'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///       as-is if the time within the CDA document contains a date-only
        ///       value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not
        ///        padded with zeroes to increase precision if the time within
        ///        the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_09()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_034());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 10
        /// Related Test Plan Test No.
        ///     35
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154
        ///     DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117
        ///     DEXS-L118 DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 DEXS-T55
        ///     DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 DEXS-T62
        ///     DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T133 DEXS-T138 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.05 UPLOAD.06
        ///     CLD.06 CLD.07 CLD.10 CLD.11 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Discharge Summary document. 
        /// Expected Result
        ///     - The document is submitted successfully with the input data
        ///       field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     -  Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     -  WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     -  Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the 
        ///       XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template
        ///       ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///       Field Name of Table 3 in the  PCEHR Document Exchange Service
        ///       TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the
        ///       value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to
        ///       the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the
        ///       value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///       identical with the value provided within the 
        ///       XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the
        ///       value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been
        ///       converted to UTC if the time within the CDA document contains
        ///       a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in
        ///       the same folder as the CDA_ROOT.xml for the purpose of
        ///       sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '18842-5'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Discharge Summary'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the admission date-time (encompassingEncounter/effectiveTime)
        ///     - The XDSDocumentEntry.serviceStopTime is set to the discharge date-time (encompassingEncounter/effectiveTime)
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)
        ///     - The Title slot is present and set to 'Discharge Summary'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///       as-is if the time within the CDA document contains a date-only
        ///       value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not
        ///        padded with zeroes to increase precision if the time within
        ///        the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_10()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_035());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 12
        /// Related Test Plan Test No.
        ///     37
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 DEXS-L102 DEXS-L103 DEXS-L104
        ///     DEXS-L105 DEXS-L153 DEXS-L154 DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8
        ///     DEXS-T9 DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94 DEXS-T95
        ///     DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125  DocumentMetadata DEXS-L106 DEXS-L107
        ///     DEXS-L108 DEXS-L109 DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 
        ///     DEXS-L118  DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 DEXS-T55 DEXS-T56 DEXS-T57
        ///     DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 DEXS-T62 DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67
        ///     DEXS-T69 DEXS-T103 DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 DEXS-T126
        ///     DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132 DEXS-T134 DEXS-T139 DEXS-T143 DEXS-T144
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.03 UPLOAD.04
        ///     CLD.06 CLD.07 CLD.08 CLD.09 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with valid input elements for a Specialist Letter document. 
        /// Expected Result
        ///     - The document is submitted successfully with the input data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     -  Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     -  WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///        Field Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located
        ///        in the same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '51852-2'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Specialist Letter'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the Document Creation Time (XDSDocumentEntry.creationTime)
        ///     - The XDSDocumentEntry.serviceStopTime is set to the Document Creation Time (XDSDocumentEntry.creationTime)
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)
        ///     - The Title slot is present and set to "Specialist Letter"
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a
        ///        date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not
        ///        padded with zeroes to increase precision if the time within
        ///        the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_12()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_037());
        }

        /// <summary>
        /// Operations
        ///     getDocument
        /// Test No.
        ///     ClientTypeChange - 13
        /// Related Test Plan Test No.
        ///     42
        /// Conformance Point
        ///     DEXS-L14 DEXS-L15 DEXS-L16 DEXS-L122 DEXS-L123 DEXS-T14 DEXS-T15 DEXS-T16
        ///     DEXS-T17 DEXS-T18 DEXS-T19 DEXS-T20 DEXS-T21 DEXS-T22 DEXS-T69 DEXS-T70
        ///     DEXS-T71
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     To ensure that the client system can execute a retrieve
        ///     document operation successfully. 
        /// Expected Result
        ///     The request should be transmitted successfully to the PCEHR
        ///     System. Upon successful execution, the requested document is
        ///     received from the PCEHR system with the ResponseStatusType:
        ///     SUCCESS in the PCEHR Header. 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_13()
        {
            this.RunTest<NOC_GetDocument>(test => test.PCEHR_NOC_042());
        }

        /// <summary>
        /// Operations	
        ///     removeDocument
        /// Test No.
        ///     ClientTypeChange - 14
        /// Related Test Plan Test No.
        ///     46
        /// Conformance Point
        ///     DEXS-L44 (CIS) DEXS-L45 (CIS) DEXS-L46 (CIS) DEXS-L47 (CIS) DEXS-L100 (CIS) DEXS-L127 (CIS)
        ///     DEXS-L64 (CRP) DEXS-L65 (CRP) DEXS-L66 (CRP) DEXS-L100 (CRP)
        ///     DEXS-T28 DEXS-T29 DEXS-T30 DEXS-T31 DEXS-T32 DEXS-T33 DEXS-T67 DEXS-T68 DEXS-T69
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or
        ///     Contracted Service Provider (CSP) or
        ///     Conformant Repository Provider (CRP)
        /// Objective/Input
        ///     Perform a Remove or deregister document operation to remove a 
        ///     document from the PCEHR system. This test case is performed by
        ///     using the document id provided in the test data package.
        /// Expected Result
        ///     A. The removeDocument request is sent successfully by using the
        ///         XDSDocumentEntry.uniqueId of the document to be removed.
        ///     B. The operation request includes the full Document Exchange Common Header.
        ///     C. The reasonForRemoval field is set to either 'IncorrectIdentity' or 'Withdrawn'.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_14()
        {
            this.RunTest<NOC_RemoveDocument>(test => test.PCEHR_NOC_046_DS());
        }

        /// <summary>
        /// Operations
        ///     For All Document Exchange Services
        /// Test No.
        ///     ClientTypeChange - 15
        /// Related Test Plan Test No.
        ///     55
        /// Conformance Point	
        ///     DEXS-L91 DEXS-L94 DEXS-L99 DEXS-L100 DEXS-L134 DEXS-L135 DEXS-L136 DEXS-L137
        ///     DEXS-L138 DEXS-L139 DEXS-L140 DEXS-L141 DEXS-L142 DEXS-L143 DEXS-L144 DEXS-L145
        ///     DEXS-L146 DEXS-L147 DEXS-L148 DEXS-L149 DEXS-L150 DEXS-L151 DEXS-L152
        ///     DEXS-L94 (CRP)
        ///     DEXS-L99 (CIS)
        ///     DEXS-T1 DEXS-T2 DEXS-T3 DEXS-T4 DEXS-T25 DEXS-T75 DEXS-T76 DEXS-T77 DEXS-T78
        ///     DEXS-T79 DEXS-T80 DEXS-T81 DEXS-T82 DEXS-T90 DEXS-T91 DEXS-T106 DEXS-T107
        ///     DEXS-T108 DEXS-T109 DEXS-T110 DEXS-T111 DEXS-T112 DEXS-T124
        /// Test Type
        ///     Positive
        /// Role	
        ///     Clinical Information System (CIS) or
        ///     Contracted Service Provider (CSP) or 
        ///     Conformant Repository Provider (CRP)
        /// Objective/Input	
        ///     Ensure the client system includes PCEHR Common Header Elements
        ///     and Transmission Signature containing a signed attestation of
        ///     elements contained within the SOAP message on all SOAP Requests.
        ///     - For this test case, the client system uses valid HPI-O
        ///       certificates provided in the test data package. Service
        ///       Invoker and Service Provider SHALL establish a secure
        ///       communication channel when interacting with each other. 
        ///     - Implementation confirms to web services base profile and TLS
        ///       security profile.
        ///     - Implementation  does not rely on support for IHE interactions.
        ///     - Unique message ID  is used in all transactions.
        ///     - IHI number is specified in all transactions, is valid, matches
        ///       the individual who owns the PCEHR and is numerical.
        ///     - ID is specified in all transactions, has no leading or
        ///       trailing spaces. 
        ///     - If ID type is HPII, the ID specified is valid and numerical.
        ///     - If ID type is LocalSystemIdentifier, the ID specified is a
        ///       representation of the access credential used to access the 
        ///       system originating the request.
        ///     - If role and/or alternate organisation is supplied, no leading
        ///       or trailing spaces used                     
        ///     - User name, vendor, product name, production version,
        ///       platform, organisation ID, organisation name,  has no leading
        ///       or trailing space, it is not blank.
        ///     - Organisation name is set to the name of the organisation.
        ///     - Product name, product version, product type and product
        ///       platform are set to the appropriate values as per client
        ///       system.
        ///     - Non-latin characters and digits are not used in any transaction
        /// Expected Result
        ///     The requests are transmitted successfully to the PCEHR System.
        ///     Response is PCEHR_SUCCESS (SUCCESS).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_15()
        {
            this.RunTest<NOC_AllDocumentExchangeServices>(test => test.PCEHR_NOC_055());
        }

        /// <summary>
        /// Operations	
        ///     uploadDocument  (Supersede)
        /// Test No.
        ///     ClientTypeChange - 19
        /// Related Test Plan Test No.
        ///     63
        /// Conformance Point	
        ///     DEXS-L120
        ///     DEXS-T67
        ///     DEXS-T68
        ///     DEXS-T69
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     To show that the client system is able to replace a document
        ///     that exists in the PCEHR System. The client system is able to
        ///     submit a request with the document association type set to a
        ///     value of ‘urn:ihe:iti:2007:AssociationType:RPLC’ (Replace).
        ///     Document being amended can be specified using a UUID or an OID.
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR System. 
        ///     Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_19()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_063_DR());
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_063_DS());
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_063_ES());
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_063_PR());
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_063_SHS());
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_063_SL());
        }

        /// <summary>
        /// Operations	
        ///     RegisterPCEHR
        /// Test No.
        ///     ClientTypeChange - 22
        /// Related Test Plan Test No.	80
        /// Conformance Point
        ///     REG-L1
        ///     REG-L2
        ///     REG-L8
        ///     REG-L16
        ///     REG-L17
        ///     REG-L18
        ///     REG-L53
        ///     REG-L54
        ///     REG-L62 (CSP)
        ///     REG-L63 (CSP)
        ///     REGS-T7
        ///     REGS-T10
        ///     REGS-T11
        ///     REGS-T8
        ///     REGS-T36
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a RegisterPCEHR operation with valid input elements.
        ///     Ensure the individual IHI Number is used in the PCEHRHeader
        ///     when the individual demographics are not used.
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR System.
        ///     Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_22()
        {
            this.RunTest<NOC_RegisterPcehr>(test => test.PCEHR_NOC_080());
        }

        /// <summary>
        /// Operations
        ///     RegisterPCEHR
        /// Test No.
        ///     ClientTypeChange - 23
        /// Related Test Plan Test No.
        ///     85
        /// Conformance Point
        ///     REG-L1
        ///     REG-L3
        ///     REG-L8
        ///     REG-L16
        ///     REG-L18
        ///     REG-L53
        ///     REG-L54
        ///     REG-L62 (CSP)
        ///     REG-L63 (CSP)
        ///     REGS-T7
        ///     REGS-T8
        ///     REGS-T14
        ///     REGS-T16
        ///     REGS-T18
        ///     REGS-T36
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a RegisterPCEHR operation with valid input elements to register a dependant.
        ///     Ensure the dependant IHI Number is used, when dependant demographics are not provided in the request.
        ///     Ensure the parental assertion flag is not used, when the dependant is on same medicare card as the representative.
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR System. Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_23()
        {
            this.RunTest<NOC_RegisterPcehr>(test => test.PCEHR_NOC_085());
        }

        /// <summary>
        /// Operations
        ///     For all RegisterPCEHR 
        /// Test No.
        ///     ClientTypeChange - 24
        /// Related Test Plan Test No.
        ///     97
        /// Conformance Point	
        ///     Reg-L49 Reg-L50 Reg-L51 Reg-L52 Reg-L53 Reg-L54 Reg-L56 Reg-L57 Reg-L58 Reg-L59 Reg-L60
        ///     Reg-L61 Reg-L62 Reg-L63 Reg-L76 Reg-L77 Reg-L78 Reg-L79 Reg-L80 Reg-L81 Reg-L82 Reg-L83
        ///     Reg-L84 Reg-L85 Reg-L86 Reg-L87 Reg-L88 Reg-L89 Reg-L90 Reg-L91 
        ///     REGS-T1 REGS-T2 REGS-T3 REGS-T4 REGS-T5 REGS-T6 REGS-T12 REGS-T18 REGS-T20 REGS-T21 
        ///     REGS-T22 REGS-T23 REGS-T24 REGS-T25 REGS-T26 REGS-T27 REGS-T28 REGS-T29 REGS-T30 
        ///     REGS-T31 REGS-T32 REGS-T33 REGS-T34 REGS-T36
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Ensure the client system includes PCEHR Common Header Elements 
        ///     and Transmission Signature containing a signed attestation of 
        ///     elements contained within the SOAP message on all SOAP Requests.
        ///     - For this test case, the client system uses valid HPI-O certificates
        ///       provided in the test data package. Service Invoker and Service
        ///       Provider SHALL establish a secure communication channel when
        ///       interacting with each other.
        ///     - Implementation confirms to web services base profile and TLS security profile.
        ///     - Unique message ID  is used in all transactions.
        ///     - IHI number is specified in all transactions, is valid, matches
        ///       the individual who owns the PCEHR and is numerical.
        ///     - ID is specified in all transactions, has no leading or trailing spaces. 
        ///     - If ID type is HPII, the ID specified is valid and numerical.
        ///     - If ID type is LocalSystemIdentifier, the ID specified is a
        ///       representation of the access credential used to access the 
        ///       system originating the request.
        ///     - If role and/or alternate organisation is supplied, no leading
        ///       or trailing spaces used                     
        ///     - User name, vendor, product name, production version, platform,
        ///       organisation ID, organisation name,  has no leading or trailing
        ///       space, it is not blank.
        ///     - Organisation name is set to the name of the organisation.
        ///     - Product name, product version, product type and product platform
        ///       are set to the appropriate values as per client system.
        /// Expected Result
        ///     The requests are transmitted successfully to the PCEHR System. 
        ///     Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_24()
        {
            this.RunTest<NOC_RegisterPcehr>(test => test.PCEHR_NOC_097());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 25
        /// Related Test Plan Test No.
        ///     112
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154
        ///     DEXS-L155 DEXS-L156 DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125
        ///     DocumentMetadata
        ///     DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109 DEXS-L112 DEXS-L113 
        ///     DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 DEXS-L118  DEXS-T50 
        ///     DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 DEXS-T55 DEXS-T56 DEXS-T57
        ///     DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 DEXS-T62 DEXS-T63 DEXS-T64
        ///     DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103 DEXS-T104 DEXS-T105 
        ///     DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 DEXS-T126 DEXS-T128
        ///     DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132 DEXS-T135 DEXS-T140
        ///     DEXS-T143 DEXS-T144  
        ///     UPLOAD.01 UPLOAD.02 
        ///     CLD.06 CLD.07 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a eHealth Prescription Record
        /// Expected Result	
        ///     - The document is submitted successfully with the input
        ///        data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the 
        ///        XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template
        ///       ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b Field
        ///        Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the
        ///        value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to
        ///        the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the
        ///        value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the 
        ///        value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been 
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in the
        ///        same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches
        ///        the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the 
        ///        author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '100.16764'
        ///     - The XDSDocumentEntry.classCodeDisplayName and
        ///        XDSDocumentEntry.typeCodeDisplayName are set to 'eHealth Prescription Record'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the author time (author/Time)
        ///     - The XDSDocumentEntry.serviceStopTime is set to the author time (author/Time)
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1) -
        ///     - The Title slot is present and set to 'eHealth Prescription Reocrd'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1) -
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not
        ///        padded with zeroes to increase precision if the time within
        ///        the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_25()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_112());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 26
        /// Related Test Plan Test No.
        ///     113
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154
        ///     DEXS-L155 DEXS-L156 DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125
        ///     DocumentMetadata
        ///     DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109 DEXS-L112 DEXS-L113 
        ///     DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 DEXS-L118  DEXS-T50 
        ///     DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 DEXS-T55 DEXS-T56 DEXS-T57
        ///     DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 DEXS-T62 DEXS-T63 DEXS-T64
        ///     DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103 DEXS-T104 DEXS-T105 
        ///     DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 DEXS-T126 DEXS-T128
        ///     DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132 DEXS-T135 DEXS-T140
        ///     DEXS-T143 DEXS-T144  
        ///     UPLOAD.01 UPLOAD.02 
        ///     CLD.06 CLD.07 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a eHealth Dispense Record
        /// Expected Result	
        ///     - The document is submitted successfully with the input
        ///        data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the 
        ///        XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template
        ///       ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b Field
        ///        Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the
        ///        value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to
        ///        the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the
        ///        value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the 
        ///        value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been 
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in the
        ///        same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches
        ///        the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the 
        ///        author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '100.16765'
        ///     - The XDSDocumentEntry.classCodeDisplayName and
        ///        XDSDocumentEntry.typeCodeDisplayName are set to 'eHealth Dispense Record'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the author time (author/Time)
        ///     - The XDSDocumentEntry.serviceStopTime is set to the author time (author/Time)
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1) -
        ///     - The Title slot is present and set to 'eHealth Dispense Record'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1) -
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not
        ///        padded with zeroes to increase precision if the time within
        ///        the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_26()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_113());
        }

        /// <summary>
        /// Operations
        ///     getView
        /// Test No.
        ///     ClientTypeChange - 27
        /// Related Test Plan Test No.
        ///     126
        /// Conformance Point
        ///     VIEW-L1
        ///     VIEW-L2
        ///     VIEW-L3
        ///     VIEW-L4
        ///     VIEW-L56
        ///     VIEW-T5
        ///     VIEW-T7
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a getView operation with valid input elements for a Prescription and Dispense View. 
        /// Expected Result
        ///     The request is transmitted successfully to the PCEHR System.
        ///     The client system is able to handle the response Status code
        ///     PCEHR_SUCCESS ( response Status description SUCCESS)
        ///     A. The individual IHI Number is provided. 
        ///     B. The version number of the corresponding view is provided.
        ///     C. The fromDate and toDate are provided.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_27()
        {
            this.RunTest<NOC_GetView>(test => test.PCEHR_NOC_126());
        }

        /// <summary>
        /// Operations
        ///     Any service operation
        /// Test No.
        ///     ClientTypeChange - 31
        /// Related Test Plan Test No.
        ///     8, 13, 15, 20, 30, 48, 66, 94, 106, 108, 132
        /// Conformance Point
        ///     8, 13, 15, 20, 30, 48, 66, 94, 106, 108, 132
        /// Test Type
        ///     Service fault
        /// Role
        ///     Negative
        /// Objective/Input
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Expected Result	
        ///     To show the ability of the client system to handle a Standard
        ///     Error (PCEHR_ERROR_0505 - Invalid HPI-O, PCEHR_ERROR_0004 - 
        ///     Authorisation denied or similar standard error) when the user
        ///     performs a  PCEHR operation. The client system is configured
        ///     with an invalid HPI-O.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_31()
        {
            this.RunTest<NOC_GetChangeHistoryView>(test => test.PCEHR_NOC_008());
            this.RunTest<NOC_GetDocumentList>(test => test.PCEHR_NOC_015());
            this.RunTest<NOC_DoesPcehrExist>(test => test.PCEHR_NOC_020());
            this.RunTest<NOC_GainPcehrAccess>(test => test.PCEHR_NOC_030());
            this.RunTest<NOC_RemoveDocument>(test => test.PCEHR_NOC_048());
            this.RunTest<NOC_RegisterPcehr>(test => test.PCEHR_NOC_094());
            this.RunTest<NOC_GetView>(test => test.PCEHR_NOC_132());

            // These services are not supported by HIPS:
            ////this.RunTest<NOC_GetAuditView>(test => test.PCEHR_NOC_013());
            ////this.RunTest<NOC_GetRepresentativeList>(test => test.PCEHR_NOC_066());
            ////this.RunTest<NOC_GetTemplate>(test => test.PCEHR_NOC_106());
            ////this.RunTest<NOC_SearchTemplate>(test => test.PCEHR_NOC_108());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 35
        /// Related Test Plan Test No.
        ///     140
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154 
        ///     DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94 
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125 
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109 
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 
        ///     DEXS-L118  DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 
        ///     DEXS-T55 DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 
        ///     DEXS-T62 DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T147 DEXS-T150 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.03 UPLOAD.04 
        ///     CLD.06 CLD.07 CLD.08 CLD.09 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Diagnostic Imaging Report document. 
        /// Expected Result	
        ///     - The document is submitted successfully with the input data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header 
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///        Field Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been 
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in the
        ///        same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '100.16957'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Diagnostic Imaging Report'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the latest imaging date recorded
        ///     - The XDSDocumentEntry.serviceStopTime is set to the latest imaging date recorded
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)- 
        ///     - The Title slot is present and set to 'Diagnostic Imaging Report'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)- 
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not padded
        ///        with zeroes to increase precision if the time within the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_35()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_140());
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     ClientTypeChange - 36
        /// Related Test Plan Test No.
        ///     141
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154 
        ///     DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94 
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125 
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109 
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 
        ///     DEXS-L118  DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 
        ///     DEXS-T55 DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 
        ///     DEXS-T62 DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T147 DEXS-T150 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.03 UPLOAD.04 
        ///     CLD.06 CLD.07 CLD.08 CLD.09 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Pathology Report document. 
        /// Expected Result	
        ///     - The document is submitted successfully with the input data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header 
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///        Field Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been 
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in the
        ///        same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '100.32001'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Pathology Report'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the latest imaging date recorded
        ///     - The XDSDocumentEntry.serviceStopTime is set to the latest imaging date recorded
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)- 
        ///     - The Title slot is present and set to 'Pathology Report'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)- 
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not padded
        ///        with zeroes to increase precision if the time within the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_ClientTypeChange")]
        public void PCEHR_NOC_CTC_36()
        {
            this.RunTest<NOC_UploadDocument>(test => test.PCEHR_NOC_141());
        }

        /// <summary>
        /// Creates an instance of a specified test class, 
        /// gives it our test context, initialises it, runs
        /// the given test method, and finishes the test report.
        /// </summary>
        /// <typeparam name="T">The test type.</typeparam>
        /// <param name="testMethod">The test method.</param>
        private void RunTest<T>(Action<T> testMethod) where T : CcaTest, new()
        {
            T test = new T();
            test.TestContext = this.TestContext;
            test.SkipHeader();
            testMethod.Invoke(test);
            var sb = test.FinishTestReport();
            this.AppendReport(sb);
        }
    }
}