﻿// -----------------------------------------------------------------------
// <copyright file="NOC_AllRecordAccessServices.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System.IO;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentProduction
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  For All Record Access Services
    /// </summary>
    [TestClass]
    public class NOC_AllRecordAccessServices : CcaTest
    {
        /// <summary>
        /// Test No:         32
        /// Objective/Input:
        ///
        /// Ensure the client system includes PCEHR Common Header Elements and
        /// Transmission Signature containing a signed attestation of elements
        /// contained within the SOAP message on all SOAP Requests.
        ///
        /// - For this test case, the client system uses valid HPI-O certificates
        ///   provided in the test data package. Service Invoker and Service
        ///   Provider SHALL establish a secure communication channel when
        ///   interacting with each other
        ///
        /// - Implementation confirms to web services base profile and TLS
        ///   security profile.
        ///
        /// - Unique message ID  is used in all transactions.
        ///
        /// - IHI number is specified in all transactions, is valid, matches
        ///   the individual who owns the PCEHR and is numerical.
        ///
        /// - ID is specified in all transactions, has no leading or trailing
        ///   spaces. If ID is HPII, ensure it is valid and numerical.
        ///
        /// - If role and/or alternate organisation is supplied, no leading
        ///   or trailing spaces used
        ///
        /// - User name, vendor, product name, production version, platform,
        ///   organisation ID, organisation name,  has no leading or trailing
        ///   space, it is not blank.
        ///
        /// - Organisation name is set to the name of the organisation.
        ///
        /// - Product name, product version, product type and product platform
        ///   are set to the appropriate values as per client system.
        ///
        /// - Service Invoker and Service Provider System SHALL comply with
        ///   audit requirements as per section 5.3 (Data Handling) in the
        ///   National E-Health Security and Access Framework [NeSAF].
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Confirmation that the payload is well formed and complies with
        /// doesPCEHRExist Request XML Schema
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_032()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            PcehrHeaderTests headerTests = new PcehrHeaderTests(this);
            headerTests.CheckHeader(GetDoesPcehrExistRequest(user, identifier), DialogueResource.HipsServiceIsPcehrAdvertised);
            headerTests.CheckHeader(GetGainAccessEmergencyRequest(user, identifier), DialogueResource.HipsServiceGainAccessEmergency);
            headerTests.CheckHeader(GetGainAccessWithoutCodeRequest(user, identifier), DialogueResource.HipsServiceGainAccessWithoutCode);
            headerTests.CheckHeader(GetGainAccessWithCodeRequest(user, identifier), DialogueResource.HipsServiceGainAccessWithCode);
        }

        private XmlDocument GetGainAccessEmergencyRequest(UserDetails user, PatientIdentifierBase identifier)
        {
            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GainEmergencyAccess, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }

        private XmlDocument GetGainAccessWithoutCodeRequest(UserDetails user, PatientIdentifierBase identifier)
        {
            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessWithOutCode(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GainAccessWithoutCode, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }

        private XmlDocument GetGainAccessWithCodeRequest(UserDetails user, PatientIdentifierBase identifier)
        {
            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessWithCode(
                identifier, patient.TestPatientRecordAccessCode, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GainAccessWithCode, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Put the access code back into the SOAP request so that the signature validates
            audit.Request = audit.Request.Replace("redacted", patient.TestPatientRecordAccessCode);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }

        private XmlDocument GetDoesPcehrExistRequest(UserDetails user, PatientIdentifierBase identifier)
        {
            DoesPcehrExistResponse pcehrResponse = ProxyHelper.PcehrProxy.IsPcehrAdvertised(identifier, patient.TargetPatient.DateOfBirth, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.DoesPcehrExist, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }
    }
}