﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using JetBrains.Annotations;
using Nehta.HL7.CDA;
using Nehta.VendorLibrary.Common;

namespace Nehta.VendorLibrary.CDA.SCSModel.Common
{
    /// <summary>
    /// This class is designed to encapsulate the properties within a CDA document that make up 
    /// a medical history,
    /// 
    /// This class implements sevearl interfaces and can be cast into these interfaces to contstrain
    /// the medical histroy to a particular type of CDA medical history
    /// 
    /// Please use the CreateMedicalHistory() method on the appropriate parent SCS object to 
    /// instantiate this class.
    /// </summary>
    [Serializable]
    [DataContract]
    internal class MedicalHistory : IMedicalHistory
    {
        #region Properties
        
        /// <summary>
        /// A list of problem diagnosis
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<IProblemDiagnosis> ProblemDiagnosis { get; set; }

        /// <summary>
        /// A list of procedures
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<Procedure> Procedures { get; set; }

        /// <summary>
        /// A list of other medical history items that contain any relevant medical history
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<IMedicalHistoryItem> MedicalHistoryItems { get; set; }

        /// <summary>
        /// Provide a custom Narrative 
        /// </summary>
        [CanBeNull]
        [DataMember]
        public StrucDocText CustomNarrativeMedicalHistory { get; set; }

        #endregion

        #region Constructors
        internal MedicalHistory()
        {
        }
        #endregion

        #region Validation

       /// <summary>
        /// Validates this medical history
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        void IMedicalHistory.Validate(string path, List<ValidationMessage> messages) 
        {
            var vb = new ValidationBuilder(path, messages);

            if (ProblemDiagnosis != null)
            {
                for (var x = 0; x < ProblemDiagnosis.Count; x++)
                {
                    ProblemDiagnosis[x].Validate(
                        vb.Path + string.Format("ProblemDiagnosis[{0}]", x), vb.Messages);
                }
            }

            if (Procedures != null)
            {
                for (var x = 0; x < Procedures.Count; x++)
                {
                    Procedures[x].Validate(
                        vb.Path + string.Format("Procedures[{0}]", x), vb.Messages);
                }
            }

            if (MedicalHistoryItems != null)
            {
                for (var x = 0; x < MedicalHistoryItems.Count; x++)
                {
                    MedicalHistoryItems[x].Validate(
                        vb.Path + string.Format("OtherMedicalHistory[{0}]", x), vb.Messages);
                }
            }
        }

        #endregion
    }

}