﻿/*
 * Copyright 2011 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.ServiceModel.Channels;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;

using Nehta.VendorLibrary.Common;
using nehta.mcaR3.ConsumerSearchIHI;

namespace Nehta.VendorLibrary.HI
{
    /// <summary>
    /// An implementation of a client for the Medicare Healthcare Identifiers service. This class may be used to 
    /// connect to Medicare's service to perform IHI searches.
    /// </summary>
    public class ConsumerSearchIHIClient : IDisposable
    {
        internal ConsumerSearchIHIPortType searchIhiClient;

        /// <summary>
        /// SOAP messages for the last client call.
        /// </summary>
        public HIEndpointProcessor.SoapMessages SoapMessages { get; set; }

        /// <summary>
        /// The ProductType to be used in all IHI searches.
        /// </summary>
        ProductType product;

        /// <summary>
        /// The User to be used in all IHI searches.
        /// </summary>
        QualifiedId user;

        /// <summary>
        /// The hpio of the organisation.
        /// </summary>
        QualifiedId hpio;

        /// <summary>
        /// Gets the timestamp for the soap request.
        /// </summary>
        public TimestampType LastSoapRequestTimestamp { get; private set; }

        /// <summary>
        /// HI service name.
        /// </summary>
        public const string HIServiceOperation = "ConsumerSearchIHI";

        /// <summary>
        /// HI service version.
        /// </summary>
        public const string HIServiceVersion = "3.0";

        #region Constructors

        /// <summary>
        /// Initializes an instance of the ConsumerSearchIHIClient.
        /// </summary>
        /// <param name="endpointUri">Web service endpoint for Medicare's consumer IHI search service.</param>
        /// <param name="product">PCIN (generated by Medicare) and platform name values.</param>
        /// <param name="user">Identifier for the application that is calling the service.</param>
        /// <param name="signingCert">Certificate to sign the soap message with.</param>
        /// <param name="tlsCert">Certificate for establishing TLS connection to the HI service.</param>
        public ConsumerSearchIHIClient(Uri endpointUri, ProductType product, QualifiedId user, QualifiedId hpio, X509Certificate2 signingCert, X509Certificate2 tlsCert)
        {
            Validation.ValidateArgumentRequired("endpointUri", endpointUri);

            InitializeClient(endpointUri.ToString(), null, signingCert, tlsCert, product, user, hpio);
        }

        /// <summary>
        /// Initializes an instance of the ConsumerSearchIHIClient.
        /// </summary>
        /// <param name="endpointConfigurationName">Endpoint configuration name for the ConsumerSearchIHI endpoint.</param>
        /// <param name="product">PCIN (generated by Medicare) and platform name values.</param>
        /// <param name="user">Identifier for the application that is calling the service.</param>
        /// <param name="signingCert">Certificate to sign the soap message with.</param>
        /// <param name="tlsCert">Certificate for establishing TLS connection to the HI service.</param>
        public ConsumerSearchIHIClient(string endpointConfigurationName, ProductType product, QualifiedId user, QualifiedId hpio, X509Certificate2 signingCert, X509Certificate2 tlsCert)
        {
            Validation.ValidateArgumentRequired("endpointConfigurationName", endpointConfigurationName);

            InitializeClient(null, endpointConfigurationName, signingCert, tlsCert, product, user, hpio);
        }

        #endregion

        /// <summary>
        /// Perform a basic search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>ihiNumber (Mandatory)</description></item>
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse BasicSearch(searchIHI request)
        {            
            Validation.ValidateArgumentRequired("request", request);
            Validation.ValidateArgumentRequired("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            Validation.ValidateArgumentNotAllowed("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentNotAllowed("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentNotAllowed("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);
            Validation.ValidateArgumentNotAllowed("request.internationalAddress", request.internationalAddress);
            Validation.ValidateArgumentNotAllowed("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentNotAllowed("request.medicareIRN", request.medicareIRN);
        
            return IHISearch(request);
        }

        /// <summary>
        /// Perform a basic medicare search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>medicareCardNumber (Mandatory)</description></item>
        /// <item><description>medicareIRN (Optional)</description></item>
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>Medicare card number</description></item>
        /// <item><description>Medicare IRN</description></item>
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse BasicMedicareSearch(searchIHI request)
        {
            Validation.ValidateArgumentRequired("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            Validation.ValidateArgumentNotAllowed("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentNotAllowed("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentNotAllowed("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentNotAllowed("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);
            Validation.ValidateArgumentNotAllowed("request.internationalAddress", request.internationalAddress);

            return IHISearch(request);
        }

        /// <summary>
        /// Perform a DVA search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>dvaFileNumber (Mandatory)</description></item>
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>DVA file number</description></item>
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse BasicDvaSearch(searchIHI request)
        {            
            Validation.ValidateArgumentRequired("request", request);
            Validation.ValidateArgumentRequired("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            Validation.ValidateArgumentNotAllowed("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentNotAllowed("request.medicareIRN", request.medicareIRN);   
            Validation.ValidateArgumentNotAllowed("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentNotAllowed("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentNotAllowed("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);
            Validation.ValidateArgumentNotAllowed("request.internationalAddress", request.internationalAddress);        

            return IHISearch(request);
        }

        /// <summary>
        /// Perform a detailed search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse DetailedSearch(searchIHI request)
        {
            Validation.ValidateArgumentRequired("request", request);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            Validation.ValidateArgumentNotAllowed("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentNotAllowed("request.medicareIRN", request.medicareIRN);
            Validation.ValidateArgumentNotAllowed("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentNotAllowed("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentNotAllowed("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentNotAllowed("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);
            Validation.ValidateArgumentNotAllowed("request.internationalAddress", request.internationalAddress);        

            return IHISearch(request);
        }

        /// <summary>
        /// Perform an australian postal address search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// <item>
        ///     <description><b>australianPostalAddress</b> (Mandatory)
        ///     <list type="bullet">
        ///         <item><description>suburb (Mandatory)</description></item>
        ///         <item><description>state (Mandatory)</description></item>
        ///         <item><description>postcode (Mandatory)</description></item>
        ///         <item>
        ///             <description><b>postalDeliveryGroup</b> (Mandatory)
        ///             <list type="bullet">
        ///                 <item><description>postalDeliveryType (Mandatory)</description></item>
        ///                 <item><description>postalDeliveryNumber (Optional)</description></item>
        ///             </list>
        ///             </description>
        ///         </item>
        ///     </list>
        ///     </description>
        /// </item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse AustralianPostalAddressSearch(searchIHI request)
        {
            Validation.ValidateArgumentRequired("request", request);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            // Check australian postal address
            Validation.ValidateArgumentRequired("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentRequired("request.australianPostalAddress.postalDeliveryGroup", request.australianPostalAddress.postalDeliveryGroup);
            Validation.ValidateArgumentRequired("request.australianPostalAddress.suburb", request.australianPostalAddress.suburb);
            Validation.ValidateArgumentRequired("request.australianPostalAddress.postcode", request.australianPostalAddress.postcode);

            Validation.ValidateArgumentNotAllowed("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentNotAllowed("request.medicareIRN", request.medicareIRN);
            Validation.ValidateArgumentNotAllowed("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentNotAllowed("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentNotAllowed("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);
            Validation.ValidateArgumentNotAllowed("request.internationalAddress", request.internationalAddress);    

            return IHISearch(request);
        }

        /// <summary>
        /// Perform an australian street address search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// <item>
        ///     <description><b>australianStreetAddress</b> (Mandatory)
        ///     <list type="bullet">
        ///         <item>
        ///             <description><b>unitGroup</b> (Optional)
        ///             <list type="bullet">
        ///                 <item><description>unitType (Mandatory)</description></item>
        ///                 <item><description>unitNumber (Optional)</description></item>
        ///             </list>
        ///             </description>
        ///         </item>
        ///         <item>
        ///             <description><b>levelGroup</b> (Optional)
        ///             <list type="bullet">
        ///                 <item><description>levelType (Mandatory)</description></item>
        ///                 <item><description>levelNumber (Optional)</description></item>
        ///             </list>
        ///             </description>
        ///         </item>
        ///         <item><description>addressSiteName (Optional)</description></item>
        ///         <item><description>streetNumber (Optional)</description></item>
        ///         <item><description>lotNumber (Optional)</description></item>
        ///         <item><description>streetName (Mandatory)</description></item>
        ///         <item><description>streetType (Conditional on if streetTypeSpecified is set to true)</description></item>
        ///         <item><description>streetTypeSpecified (Mandatory)</description></item>
        ///         <item><description>streetSuffix (Conditional on if streetSuffixSpecified is set to true)</description></item>
        ///         <item><description>streetSuffixSpecified (Mandatory)</description></item>
        ///         <item><description>suburb (Mandatory)</description></item>
        ///         <item><description>state (Mandatory)</description></item>
        ///         <item><description>postcode (Mandatory)</description></item>
        ///     </list>
        ///     </description>
        /// </item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse AustralianStreetAddressSearch(searchIHI request)
        {
            Validation.ValidateArgumentRequired("request", request);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            // Check australian street address
            Dictionary<string, object> c1 = new Dictionary<string, object>();
            c1.Add("request.australianStreetAddress.streetNumber", request.australianStreetAddress.streetNumber);
            c1.Add("request.australianStreetAddress.lotNumber", request.australianStreetAddress.lotNumber);
            Validation.ValidateArgumentAtLeastOneRequired(c1);
            Validation.ValidateArgumentRequired("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentRequired("request.australianStreetAddress.postcode", request.australianStreetAddress.postcode);
            Validation.ValidateArgumentRequired("request.australianStreetAddress.suburb", request.australianStreetAddress.suburb);
            Validation.ValidateArgumentRequired("request.australianStreetAddress.streetName", request.australianStreetAddress.streetName);

            Validation.ValidateArgumentNotAllowed("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentNotAllowed("request.medicareIRN", request.medicareIRN);
            Validation.ValidateArgumentNotAllowed("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentNotAllowed("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentNotAllowed("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);
            Validation.ValidateArgumentNotAllowed("request.internationalAddress", request.internationalAddress);

            return IHISearch(request);
        }
        
        /// <summary>
        /// Perform an international address search on the ConsumerSearchIHI service.
        /// </summary>
        /// <param name="request">
        /// The search criteria. The following fields are expected:
        /// <list type="bullet">
        /// <item><description>familyName (Mandatory)</description></item>
        /// <item><description>givenName (Optional)</description></item>
        /// <item><description>dateOfBirth (Mandatory)</description></item>
        /// <item><description>sex (Mandatory)</description></item>
        /// <item>
        ///     <description><b>internationalAddress</b> (Mandatory)
        ///     <list type="bullet">
        ///         <item><description>internationalAddressLine (Mandatory)</description></item>
        ///         <item><description>internationalStateProvince (Mandatory)</description></item>
        ///         <item><description>internationalPostcode (Mandatory)</description></item>
        ///         <item><description>country (Mandatory)</description></item>
        ///     </list>
        ///     </description>
        /// </item>
        /// </list>
        /// All other fields are to be null.
        /// </param>
        /// <returns>
        /// A searchIHIResponse instance containing:
        /// <list type="bullet">
        /// <item><description>IHI number</description></item>
        /// <item><description>IHI record status</description></item>
        /// <item><description>IHI status</description></item>
        /// <item><description>Family name</description></item>
        /// <item><description>Given name (if entered in search criteria)</description></item>
        /// <item><description>Date of birth</description></item>
        /// <item><description>Sex</description></item>
        /// </list>
        /// </returns>
        public searchIHIResponse InternationalAddressSearch(searchIHI request)
        {
            Validation.ValidateArgumentRequired("request", request);
            Validation.ValidateArgumentRequired("request.familyName", request.familyName);
            Validation.ValidateDateTime("request.dateOfBirth", request.dateOfBirth);

            // Check international address
            Validation.ValidateArgumentRequired("request.internationalAddress", request.internationalAddress);
            Validation.ValidateArgumentRequired("request.internationalAddress.internationalStateProvince", request.internationalAddress.internationalStateProvince);
            Validation.ValidateArgumentRequired("request.internationalAddress.internationalPostcode", request.internationalAddress.internationalPostcode);
            Validation.ValidateArgumentRequired("request.internationalAddress.internationalAddressLine", request.internationalAddress.internationalAddressLine);

            Validation.ValidateArgumentNotAllowed("request.medicareCardNumber", request.medicareCardNumber);
            Validation.ValidateArgumentNotAllowed("request.medicareIRN", request.medicareIRN);
            Validation.ValidateArgumentNotAllowed("request.ihiNumber", request.ihiNumber);
            Validation.ValidateArgumentNotAllowed("request.australianPostalAddress", request.australianPostalAddress);
            Validation.ValidateArgumentNotAllowed("request.australianStreetAddress", request.australianStreetAddress);
            Validation.ValidateArgumentNotAllowed("request.dvaFileNumber", request.dvaFileNumber);
            Validation.ValidateArgumentNotAllowed("request.history", request.historySpecified);  

            return IHISearch(request);
        }

        #region Private and internal methods

        /// <summary>
        /// Perform the IHI service call.
        /// </summary>
        /// <param name="request">The search criteria in a searchIHI object.</param>
        /// <returns>The IHI search results.</returns>
        private searchIHIResponse IHISearch(searchIHI request)
        {
            searchIHIRequest envelope = new searchIHIRequest();

            envelope.searchIHI = request;
            envelope.product = product;
            envelope.user = user;
            envelope.hpio = hpio;
            envelope.signature = new SignatureContainerType();

            envelope.timestamp = new TimestampType()
            {
                created = DateTime.Now.ToUniversalTime(),
                expires = DateTime.Now.AddDays(30).ToUniversalTime(),
                expiresSpecified = true
            };

            // Set LastSoapRequestTimestamp
            LastSoapRequestTimestamp = envelope.timestamp;

            searchIHIResponse1 response1 = null;

            try
            {
                response1 = searchIhiClient.searchIHI(envelope);
            }
            catch (Exception ex)
            {
                // Catch generic FaultException and call helper to throw a more specific fault
                // (FaultException<ServiceMessagesType>
                FaultHelper.ProcessAndThrowFault<ServiceMessagesType>(ex);
            }

            if (response1 != null && response1.searchIHIResponse != null)
                return response1.searchIHIResponse;
            else
                throw new ApplicationException(Properties.Resources.UnexpectedServiceResponse);
        }

        /// <summary>
        /// Initializes an instance of the ConsumerSearchIHIClient.
        /// </summary>
        /// <param name="endpointUrl">Web service endpoint for Medicare's consumer IHI search service.</param>
        /// <param name="endpointConfigurationName">Endpoint configuration name for the ConsumerSearchIHI endpoint.</param>
        /// <param name="signingCert">Certificate to sign the soap message with.</param>
        /// <param name="tlsCert">Certificate for establishing TLS connection to the HI service.</param>
        /// <param name="product">PCIN (generated by Medicare) and platform name values.</param>
        /// <param name="user">Identifier for the application that is calling the service.</param>
        /// <param name="hpio">Identifier for the organisation</param>
        private void InitializeClient(string endpointUrl, string endpointConfigurationName, X509Certificate2 signingCert, X509Certificate2 tlsCert, ProductType product, QualifiedId user, QualifiedId hpio)
        {
            Validation.ValidateArgumentRequired("product", product);
            Validation.ValidateArgumentRequired("user", user);
            Validation.ValidateArgumentRequired("signingCert", signingCert);
            Validation.ValidateArgumentRequired("tlsCert", tlsCert);

            this.product = product;
            this.user = user;
            this.hpio = hpio;

            SoapMessages = new HIEndpointProcessor.SoapMessages();

            ConsumerSearchIHIPortTypeClient client = null;

            if (!string.IsNullOrEmpty(endpointUrl))
            {
                EndpointAddress address = new EndpointAddress(endpointUrl);
                CustomBinding tlsBinding = GetBinding();

                client = new ConsumerSearchIHIPortTypeClient(tlsBinding, address);
            }
            else if (!string.IsNullOrEmpty(endpointConfigurationName))
            {
                client = new ConsumerSearchIHIPortTypeClient(endpointConfigurationName);
            }

            if (client != null)
            {
                HIEndpointProcessor.ProcessEndpoint(client.Endpoint, signingCert, SoapMessages);

                if (tlsCert != null)
                {
                    client.ClientCredentials.ClientCertificate.Certificate = tlsCert;
                }

                searchIhiClient = client;
            }
        }

        /// <summary>
        /// Gets the binding configuration for the client.
        /// </summary>
        /// <returns>Configured CustomBinding instance.</returns>
        internal CustomBinding GetBinding()
        {
            // Set up binding
            CustomBinding tlsBinding = new CustomBinding();

            TextMessageEncodingBindingElement tlsEncoding = new TextMessageEncodingBindingElement();
            tlsEncoding.ReaderQuotas.MaxDepth = 2147483647;
            tlsEncoding.ReaderQuotas.MaxStringContentLength = 2147483647;
            tlsEncoding.ReaderQuotas.MaxArrayLength = 2147483647;
            tlsEncoding.ReaderQuotas.MaxBytesPerRead = 2147483647;
            tlsEncoding.ReaderQuotas.MaxNameTableCharCount = 2147483647;

            HttpsTransportBindingElement httpsTransport = new HttpsTransportBindingElement();
            httpsTransport.RequireClientCertificate = true;
            httpsTransport.MaxReceivedMessageSize = 2147483647;
            httpsTransport.MaxBufferSize = 2147483647;

            tlsBinding.Elements.Add(tlsEncoding);
            tlsBinding.Elements.Add(httpsTransport);

            return tlsBinding;
        }

        #endregion

        #region IDisposable Members

        /// <summary>
        /// Closes and disposes the client.
        /// </summary>
        public void Dispose()
        {
            ClientBase<ConsumerSearchIHIPortType> searchClient;

            if (searchIhiClient is ClientBase<ConsumerSearchIHIPortType>)
            {
                searchClient = (ClientBase<ConsumerSearchIHIPortType>)searchIhiClient;
                if (searchClient.State != CommunicationState.Closed)
                    searchClient.Close();
            }
        }

        #endregion
    }
}
